/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.resource.ResourceException;

import org.mule.api.MuleException;
import org.mule.tools.module.invocation.DynamicModule;

import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.module.AbstractOperationHandler;
import com.mulesoft.adapter.module.PIMessageParameters;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.engine.interfaces.messaging.api.auditlog.AuditLogStatus;
import com.sforce.async.BatchInfo;
import com.sforce.async.BatchResult;
import com.sforce.soap.partner.SaveResult;

/**
 * Provide support for `create` operation.
 * 
 * Expected input format:
 * 
 * <sObjects>
 *   <type>Account</type>
 *   <Name></Name>
 * </sObjects>
 * 
 * or
 * 
 * <sObjects xsi:type="ns3:Account" xmlns:ns="urn:sobject.enterprise.soap.sforce.com">
 *   <ns:Name></ns:Name>
 * </sObjects>
 * 
 * Several sObject can be provided but they must all have the same type.
 * 
 * A single `create` group is expected.
 * 
 * @see http://www.salesforce.com/us/developer/docs/api/Content/sforce_api_calls_create.htm
 * @see http://www.salesforce.com/us/developer/resources/soap/sforce60/sforce_API_messages_create.html
 */
public class CreateOperationHandler extends AbstractOperationHandler {

    private static final String PROCESSOR_NAME = "create";
    private static final String BULK_PROCESSOR_NAME = "create-bulk";
    private static final String TYPE_PARAMETER_NAME = "type";
    private static final String OBJECTS_PARAMETER_NAME = "objects";
    private static final int MAX_RECORDS_NUMBER = 200;

    public CreateOperationHandler(final DynamicModule module) {
        super(CreateOperationHandler.PROCESSOR_NAME, module);
    }

    protected final void create(final String type, final List<Map<String, Object>> objects, IPILogger logger) throws MuleException, ResourceException {
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(CreateOperationHandler.TYPE_PARAMETER_NAME, type);
        parameters.put(CreateOperationHandler.OBJECTS_PARAMETER_NAME, objects);

        logger.reportProcessingStatus(ProcessState.OK, "invoking operation");
        final List<SaveResult> results = invoke(parameters);
        logger.reportProcessingStatus(ProcessState.OK, "invocation finished");
        Results.throwIfSaveNotSuccessful(results);
    }

    protected final void createBulk(final String type, final List<Map<String, String>> objects, IPILogger logger) throws MuleException, ResourceException, InterruptedException {
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(CreateOperationHandler.TYPE_PARAMETER_NAME, type);
        parameters.put(CreateOperationHandler.OBJECTS_PARAMETER_NAME, objects);

        logger.reportProcessingStatus(ProcessState.OK, "invoking bulk operation");
        final BatchInfo batchInfo = invoke(CreateOperationHandler.BULK_PROCESSOR_NAME, parameters);
        logger.reportProcessingStatus(ProcessState.OK, "awaiting bulk operation completion");
        final BatchResult batchResult = Bulks.awaitCompletion(getModule(), batchInfo);
        logger.reportProcessingStatus(ProcessState.OK, "invocation finished");
        Bulks.throwIfNotSuccessful(batchResult);
    }

    @Override
    public final byte[] handle(PIMessageParameters messageParameters) throws MuleException, ResourceException {
        Map<String, List<Map<String, String>>> typeObjectsMap = SObjects.parse(messageParameters.getRequestContent());
        
        if (typeObjectsMap.isEmpty()) {
            throw new IllegalArgumentException("No SObjects identified.");
        }
        if (typeObjectsMap.size() > 1) {
            throw new IllegalArgumentException("More than a single type has been sent.");
        }
        final Map.Entry<String, List<Map<String, String>>> entry = typeObjectsMap.entrySet().iterator().next();
        final String type = entry.getKey();
        final List<Map<String, String>> objects = entry.getValue();
        int size = objects.size();
        
        IPILogger logger = messageParameters.getLogger();
        logger.reportAuditStatus(AuditLogStatus.SUCCESS, "Message of type {0} contains {1} sObjects", type, size);
        
		if (size <= CreateOperationHandler.MAX_RECORDS_NUMBER) {
		    //create uses a different representation of e.g. date
		    final List<Map<String, Object>> typedTypeObjectsMap = SObjects.asTypedObject(getModule(), type, objects);
            create(type, typedTypeObjectsMap, logger);
        } else {
            try {
                createBulk(type, objects, logger);
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                return null;
            }
        }
        return null;
    }
    
}