/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.resource.ResourceException;

import org.mule.api.MuleException;
import org.mule.tools.module.invocation.DynamicModule;
import org.mule.util.StringUtils;

import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.XML;
import com.mulesoft.adapter.module.AbstractOperationHandler;
import com.mulesoft.adapter.module.PIMessageParameters;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.engine.interfaces.messaging.api.auditlog.AuditLogStatus;

/**
 * Provide support for `query` operation.
 * 
 * Expected input format:
 * 
 * <query>
 *   <queryString>select FirstName, LastName from Contact</queryString>
 * </query>
 * 
 * Output format:
 * 
 * <queryResponse xmlns="urn:enterprise.soap.sforce.com">    
 *   <result>     
 *     <done>true</done>     
 *     <queryLocator>01g30000000590JAAQ-3</queryLocator>     
 *     <records xsi:type="sf:Contact" xmlns:sf="urn:sobject.enterprise.soap.sforce.com">      
 *       <sf:FirstName>Merce</sf:FirstName>      
 *       <sf:LastName>Carroll</sf:LastName>     
 *     </records>        
 *     <size>1</size>    
 *   </result>   
 * </queryResponse>  
 * 
 * @see http://www.salesforce.com/us/developer/docs/api/Content/sforce_api_calls_query.htm
 * @see http://www.salesforce.com/us/developer/resources/soap/sforce60/sforce_API_messages_query.html
 */
public class QueryOperationHandler extends AbstractOperationHandler {

    private static final String PROCESSOR_NAME = "query";
    private static final String QUERY_PARAMETER_NAME = "query";

    public QueryOperationHandler(final DynamicModule module) {
        super(QueryOperationHandler.PROCESSOR_NAME, module);
    }

    @Override
    public final byte[] handle(PIMessageParameters messageParameters) throws MuleException, ResourceException {
        final List<List<String>> queriesGroups = XML.textNodes(messageParameters.getRequestContent());
        if (queriesGroups.isEmpty()) {
            throw new IllegalArgumentException("No queryGroup identified.");
        }
        if (queriesGroups.size() >1) {
            throw new IllegalArgumentException("More than a single query group has been sent.");
        }
        final String query = queriesGroups.get(0).get(0).trim();
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(QueryOperationHandler.QUERY_PARAMETER_NAME, query);
        
        IPILogger logger = messageParameters.getLogger();
        logger.reportAuditStatus(AuditLogStatus.SUCCESS, "Message contains query \"{0}\"", StringUtils.abbreviate(query, 250)); 

        logger.reportProcessingStatus(ProcessState.OK, "invoking query operation");
        final List<Map<String, Object>> results = invoke(parameters);
        logger.reportProcessingStatus(ProcessState.OK, "invocation finished, query returned {0} results", results.size());
        
        return SObjects.generateQueryResponse(results, messageParameters.getResponseRootElementName(), messageParameters.getResponseRootElementNamespace());
    }
    
}