/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.adapter.module.salesforce;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import javax.resource.ResourceException;

import org.mule.api.MessagingException;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.modules.salesforce.config.SalesforceModuleConnectionManager;
import org.mule.modules.salesforce.config.SalesforceModuleLifecycleAdapter;
import org.mule.tools.module.loader.Loader;
import org.mule.tools.module.model.Module;
import org.mule.util.StringUtils;

import com.mulesoft.adapter.helper.Channels;
import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.module.AbstractPIModule;
import com.mulesoft.adapter.ra.XIMessageFactoryImpl;
import com.sap.aii.af.lib.mp.processor.ModuleProcessorException;
import com.sap.aii.af.service.administration.api.monitoring.ProcessState;
import com.sap.aii.af.service.cpa.Channel;
import com.sforce.soap.partner.fault.UnexpectedErrorFault;

/**
 * A concrete {@link AbstractPIModule} for Salesforce.
 * Rely on Mule <a href="https://github.com/mulesoft/salesforce-connector">Salesforce connector</a>.
 * <br>
 * Currently offers following operations:
 * <ul>
 *   <li>query</li>
 *   <li>create</li>
 *   <li>upsert</li>
 *   <li>delete</li>
 * </ul>
 */
public class SalesforcePIModule extends AbstractPIModule {

    static class SalesforceProxySelector extends ProxySelector {

        private final ProxySelector defaultProxySelector;
        private final String host;
        private final int port;

        public SalesforceProxySelector(final ProxySelector defaultProxySelector, final String host, final int port) {
            this.host = host;
            this.port = port;
            this.defaultProxySelector =defaultProxySelector;
        }

        @Override
        public List<Proxy> select(final URI uri) {
            try {
                if (uri.equals(new URI(this.host))) {
                    return Collections.singletonList(new Proxy(Proxy.Type.HTTP, new InetSocketAddress(this.host, port)));
                }
            } catch (URISyntaxException e) {
                SalesforcePIModule.LOCATION.warningT("Failed to parse url <"+this.host+">");
                SalesforcePIModule.LOCATION.throwing(e);
            }

            return this.defaultProxySelector.select(uri);
        }

        @Override
        public void connectFailed(final URI uri, final SocketAddress sa, final IOException e) {
            SalesforcePIModule.LOCATION.warningT("Failed to connect to <"+uri+">");
            SalesforcePIModule.LOCATION.throwing(e);
        }

    }

    private static final String PROXY_HOST_PARAMETER_NAME = "proxyHost";
    private static final String PROXY_PORT_PARAMETER_NAME = "proxyPort";
    private static final String PROXY_USERNAME_PARAMETER_NAME = "proxyUsername";
    private static final String PROXY_PASSWORD_PARAMETER_NAME = "proxyPassword";
    private static final String TOPIC_PARAMETER_NAME = "topic";
    private static final String SUBSCRIBE_TOPIC_NAME = "subscribe-topic";
    
    public SalesforcePIModule(final Channel channel) throws ResourceException {
        super(channel);

        registerSupportedOperationHandlers();
    }
    
    private void registerSupportedOperationHandlers() {
        //Register all supported OperationHandler
        register(new QueryOperationHandler(getModule()));
        register(new CreateOperationHandler(getModule()));
        register(new UpsertOperationHandler(getModule()));
        register(new DeleteOperationHandler(getModule()));
    }

    @Override
    protected final Module createModule(final Channel channel) throws ResourceException {
        final String username = Channels.retrieveUsername(channel);
        final String password = Channels.retrievePassword(channel);
        final String securityToken = Channels.retrieveSecurityToken(channel);
        Module module = new Loader().load(new SalesforceModuleLifecycleAdapter(), new SalesforceModuleConnectionManager());
        module.setUsername(username);
        module.setPassword(password);
        module.setSecurityToken(securityToken);
        return module;
    }
    
    @Override
    public boolean retryOn(final Throwable cause) {
        final String SIGNATURE = "retryOn(final Throwable cause)";
        LOCATION.entering(SIGNATURE, new Object[] {cause});
        
        boolean retry = false;
        
        if (cause instanceof MessagingException) {
            Throwable causeOfCause = cause.getCause();
            if (causeOfCause instanceof UnexpectedErrorFault) {
                final UnexpectedErrorFault exception = (UnexpectedErrorFault) causeOfCause;
                switch (exception.getExceptionCode()) {
                case REQUEST_LIMIT_EXCEEDED:
                    retry = true;
                }
            }
            else if (causeOfCause instanceof NoSuchElementException) {
                // Workaround commons-pool swallowing exceptions
                // See https://github.com/mulesoft/mule-devkit/issues/53
                retry = causeOfCause.toString().contains("TotalRequests Limit exceeded");
            }
        }
        
        LOCATION.exiting(SIGNATURE, retry);
        return retry;
    }

    @Override
    protected final Map<String, Object> createParameters(final Channel channel) throws ResourceException {
        final String proxyHost = Channels.retrieveConfigurationElement(SalesforcePIModule.PROXY_HOST_PARAMETER_NAME, channel);
        final int proxyPort = Channels.retrieveConfigurationElementAsInt(SalesforcePIModule.PROXY_PORT_PARAMETER_NAME, channel);
        final String proxyUsername = Channels.retrieveConfigurationElement(SalesforcePIModule.PROXY_USERNAME_PARAMETER_NAME, channel);
        final String proxyPassword = Channels.retrieveConfigurationElement(SalesforcePIModule.PROXY_PASSWORD_PARAMETER_NAME, channel);
        final Map<String, Object> parameterValues = new HashMap<String, Object>();
        if (Channels.isDefined(proxyHost)) {
            parameterValues.put(SalesforcePIModule.PROXY_HOST_PARAMETER_NAME, proxyHost);

            //Use specific proxy settings when salesforce url is used.
            //Do not rely on native salesforce-connector proxy support due to a Netweaver limitation.
            final ProxySelector defaultProxySelector = ProxySelector.getDefault();
            ProxySelector.setDefault(new SalesforceProxySelector(defaultProxySelector, proxyHost, (proxyPort != 0) ? proxyPort : 8080));
        }
        if (proxyPort != 0) {
            parameterValues.put(SalesforcePIModule.PROXY_PORT_PARAMETER_NAME, proxyPort);
        }
        if (Channels.isDefined(proxyUsername)) {
            parameterValues.put(SalesforcePIModule.PROXY_USERNAME_PARAMETER_NAME, proxyUsername);
        }
        if (Channels.isDefined(proxyPassword)) {
            parameterValues.put(SalesforcePIModule.PROXY_PASSWORD_PARAMETER_NAME, proxyPassword);
        }
        return parameterValues;
    }

    /**
     * Subscribe to {@linkplain SalesforcePIModule#SUBSCRIBE_TOPIC_NAME}.
     * 
     * @param messageFactory
     * @throws InitialisationException
     * @throws MuleException
     * @throws ModuleProcessorException 
     */
    public final void subscribe(final XIMessageFactoryImpl messageFactory, IPILogger logger) throws MuleException, ModuleProcessorException, ResourceException {
        if (messageFactory == null) {
            throw new IllegalArgumentException("null messageFactory");
        }

        final String topicName = Channels.retrieveConfigurationElement(SalesforcePIModule.TOPIC_PARAMETER_NAME, getChannel());
        if (topicName == null) {
            throw new IllegalArgumentException("Cannot access mandatory parameter <"+SalesforcePIModule.TOPIC_PARAMETER_NAME+">");
        }
        final Map<String, Object> parameters = new HashMap<String, Object>();
        parameters.put(SalesforcePIModule.TOPIC_PARAMETER_NAME, topicName);

        logger.reportProcessingStatus(ProcessState.OK, "subscribing to topic {0}", StringUtils.abbreviate(topicName, 120));
        
        new TransportConfigurator().configureNetweaverHttpTransport();
        
        subscribe(SalesforcePIModule.SUBSCRIBE_TOPIC_NAME, parameters, messageFactory);
        
        logger.reportProcessingStatus(ProcessState.OK, "subscribed, wating for data");
    }

    /**
     * Unsubscribe from {@linkplain SalesforcePIModule#SUBSCRIBE_TOPIC_NAME}.
     * 
     * @param sourceName
     * @throws MuleException 
     */
    public final void unsubscribe(IPILogger logger) throws MuleException {
        logger.reportProcessingStatus(ProcessState.OK, "unsubscribing");
        unsubscribe(SalesforcePIModule.SUBSCRIBE_TOPIC_NAME);
    }

}