/*
 * (C) 2006 SAP XI 7.1 Adapter Framework Resource Adapter Skeleton
 */

package com.mulesoft.adapter.ra;

import java.io.PrintWriter;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.resource.NotSupportedException;
import javax.resource.ResourceException;
import javax.resource.spi.ConnectionEventListener;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionFactory;
import javax.resource.spi.ManagedConnectionMetaData;
import javax.resource.spi.security.PasswordCredential;
import javax.security.auth.Subject;
import javax.transaction.xa.XAResource;

import com.mulesoft.adapter.helper.IPILogger;
import com.mulesoft.adapter.helper.PILogger;
import com.mulesoft.adapter.module.PIModule;
import com.mulesoft.adapter.module.salesforce.SAPHttpTransport;
import com.mulesoft.adapter.module.salesforce.SalesforcePIModule;
import com.mulesoft.adapter.module.salesforce.TransportConfigurator;
import com.sap.aii.af.service.cpa.Channel;
import com.sap.engine.interfaces.messaging.api.Message;
import com.sforce.ws.ConnectorConfig;

/**
 * A <code>SpiManagedConnection</code> represents a physical, managed connection
 * from the resource adapter to the connected external system. In this sample
 * resource adapter it represents a simple file.
 * 
 * @version: $Id:
 *           //tc/xpi.external/NW730EXT_01_REL/src/_sample_rar_module/rar/src
 *           /com/sap/aii/af/sample/adapter/ra/SPIManagedConnection.java#1 $
 **/

public class SPIManagedConnection implements ManagedConnection, Serializable {

    private static final long serialVersionUID = 1L;

    private static final XITrace TRACE = new XITrace(SPIManagedConnection.class.getName());

    private XIConnectionEventListenerManager cciListener;
    private PasswordCredential credential;
    private SPIManagedConnectionFactory mcf;
    private PrintWriter logWriter;
    private boolean supportsLocalTx;
    private boolean destroyed;
    private Set<CCIConnection> connectionSet; // Multiple CCI connections can
                                              // operate on one managed
                                              // connection. Stored in this
                                              // connectionSet
    private String channelID = null; // Corresponding XI channel ID
    private Channel channel = null; // Corresponding XI channel as CPA object
    private boolean asmaGet = false; // True if the adapter specific message
                                     // property must be read
    private boolean asmaError = false; // True if an error must be thrown when
                                       // the ASMA is not set
    private PIModule module;

    SPIManagedConnection(SPIManagedConnectionFactory managedConnectionFactory, PasswordCredential credential, boolean supportsLocalTx, String channelID,
            Channel channel) throws ResourceException, NotSupportedException {
        final String SIGNATURE = "SpiManagedConnection(ManagedConnectionFactory mcf, PasswordCredential credential, boolean supportsLocalTx, String channelID)";
        TRACE.entering(SIGNATURE, new Object[] { managedConnectionFactory, credential, Boolean.valueOf(supportsLocalTx), channelID });

        if (supportsLocalTx == true) {
            throw new NotSupportedException("Local transactions are not supported!");
        }

        this.mcf = managedConnectionFactory;
        this.credential = credential;
        this.supportsLocalTx = supportsLocalTx;
        this.channelID = channelID;
        this.channel = channel;

        connectionSet = new HashSet<CCIConnection>();
        cciListener = new XIConnectionEventListenerManager(this);

        new TransportConfigurator().configureNetweaverHttpTransport();

        try {
            this.module = new SalesforcePIModule(this.channel);
        } catch (ResourceException e) {
            TRACE.catching(SIGNATURE, e);
            TRACE.warningT(SIGNATURE, XIAdapterCategories.CONNECT_EIS, "Exception during PIModule creation. Reason: {0}", new Object[] { e.getMessage() });
            TRACE.throwing(SIGNATURE, e);
            throw e;
        }

        TRACE.exiting(SIGNATURE);
    }

    public IPILogger createLogger(Message message) {
        return new PILogger(channel, message);
    }

    public final PIModule getModule() {
        return this.module;
    }

    /**
     * Adapter specific message attribute switch getter
     * 
     * (ra implementation specific)
     * 
     * @return True if adapter specific message attribute must be evaluated
     */
    boolean getAsmaGet() {
        return asmaGet;
    }

    /**
     * Adapter specific message attribute switch error handling
     * 
     * (ra implementation specific)
     * 
     * @return True if an error must be thrown if the adapter specific message
     *         attribute is missing
     */
    boolean getAsmaError() {
        return asmaError;
    }

    /**
     * Returns the channelID of the channel that uses this managed connection In
     * this sample the relationship XI channel : managedConnection is 1:1 In
     * more sophisticated implementations the relationship probably is more n:m
     * with n>m
     * 
     * (ra implementation specific)
     * 
     * @return channelID of the channel
     */
    String getChannelID() {
        return channelID;
    }

    /**
     * Switches on the support of local transactions.
     * 
     * (ra implementation specific)
     * 
     * @param ltx
     *            True, if local transactions must be supported, else false.
     *            Must be false currently.
     * @throws NotSupportedException
     *             If supportsLocalTx equals true since local transactions are
     *             currently not supported
     */
    public void setSupportsLocalTx(boolean ltx) throws NotSupportedException {
        if (ltx == true)
            throw new NotSupportedException("Local transactions are not supported!");

        this.supportsLocalTx = ltx;
    }

    /**
     * Returns true if local transactions are supported.
     * 
     * (ra implementation specific)
     * 
     * @return true, if local transactions are supported, else false
     */
    public boolean getSupportsLocalTx() {
        return this.supportsLocalTx;
    }

    /**
     * Assignes this <code>ManagedConnection</code> to a (new)
     * <code>ManagedConnectionFactory</code>
     * 
     * (ra implementation specific)
     * 
     * @param mcf
     *            <code>ManagedConnectionFactory</code> to which this connection
     *            must be assigned to
     */
    public void setManagedConnectionFactory(SPIManagedConnectionFactory mcf) {
        this.mcf = mcf;
    }

    /**
     * Returns the associated <code>ManagedConnectionFactory</code>
     * 
     * (ra implementation specific)
     * 
     * @return Associated <code>ManagedConnectionFactory</code>
     */
    public ManagedConnectionFactory getManagedConnectionFactory() {
        return this.mcf;
    }

    /**
     * Returns a CCI connection that operates on this managed connection. The
     * <code>CciConnection</code> object is always created newly. The
     * credentials must be equal to the one which was used when the
     * <code>ManagedConnectionFactory</code> was created. (SPI JCA 1.0)
     * 
     * @return CCI connection for the XI AF to work with
     * @throws SecurityException
     *             Subject does not fit to the original specified credentials
     * @throws ResourceException
     *             CCI connection cannot be created
     * @throws IllegalStateException
     *             Managed connection is already destroyed
     */
    public Object getConnection(Subject subject, ConnectionRequestInfo info) throws ResourceException {
        final String SIGNATURE = "getConnection(Subject subject, ConnectionRequestInfo info)";
        TRACE.entering(SIGNATURE, new Object[] { subject, info });

        // Check credentials
        PasswordCredential newCredential = XISecurityUtilities.getPasswordCredential(mcf, subject, info);
        if (!XISecurityUtilities.isPasswordCredentialEqual(newCredential, credential)) {
            throw new javax.resource.spi.SecurityException("Principal does not match." + "Reauthentication not supported");
        }

        // Create CCI connection
        checkIfDestroyed();
        CCIConnection cciConnection = new CCIConnection(this);
        addCciConnection(cciConnection);
        TRACE.exiting(SIGNATURE);
        return cciConnection;
    }

    /**
     * Destroys the underlying physical connection (i.e. the output file is
     * closed) and invalidates the registered CCI connections. This managed
     * connection cannot be used further more. (SPI JCA 1.0)
     * 
     * @throws ResourceException
     *             CCI connection cannot be invalidated or file cannot be closed
     */
    public void destroy() throws ResourceException {
        final String SIGNATURE = "destroy()";
        TRACE.entering(SIGNATURE);
        destroy(false);
        TRACE.exiting(SIGNATURE);
    }

    /**
     * @see com.mulesoft.adapter.ra.SPIManagedConnection#destroy() This variant
     *      informs the MCF to remove the MC from the internal pool, if called
     *      by the JCA container (ra implementation specific)
     * 
     * @param fromMCF
     *            True if the MC destroy was initiated by the MCF, else by the
     *            JCA container
     * @throws ResourceException
     *             CCI connection cannot be invalidated or file cannot be closed
     */
    void destroy(boolean fromMCF) throws ResourceException {
        final String SIGNATURE = "destroy(boolean fromMCF)";
        TRACE.entering(SIGNATURE, new Object[] { Boolean.valueOf(fromMCF) });

        if (!destroyed) {
            try {
                destroyed = true;
                invalidateAllConnections();

                module.dispose();
                module = null;
            } catch (Exception ex) {
                TRACE.catching(SIGNATURE, ex);
                throw new ResourceException(ex.getMessage());
            }
        }

        if (!fromMCF) {
            mcf.removeManagedConnection(this.channelID);
        }

        TRACE.exiting(SIGNATURE);
    }

    /**
     * Cleanups this managed connection. The current output file is closed if
     * file mode is set to "new". It invalidates the registered CCI connections.
     * This managed connection can be be used later, but writes to another file
     * than before. (SPI JCA 1.0)
     * 
     * @throws ResourceException
     *             CCI connection cannot be invalidated or file cannot be closed
     */
    public void cleanup() throws ResourceException {
        final String SIGNATURE = "cleanup()";
        TRACE.entering(SIGNATURE);

        try {
            checkIfDestroyed();
            invalidateAllConnections();
        } catch (Exception ex) {
            TRACE.catching(SIGNATURE, ex);
            throw new ResourceException(ex.getMessage());
        }

        TRACE.exiting(SIGNATURE);
    }

    private void invalidateAllConnections() {
        Iterator<?> it = connectionSet.iterator();
        while (it.hasNext()) {
            CCIConnection cciCon = (CCIConnection) it.next();
            cciCon.invalidate();
        }
        connectionSet.clear();
    }

    /**
     * Associate a previously created CCI connection to this managed connection
     * (SPI JCA 1.0)
     * 
     * @param connection
     *            CciConnection which has to be assigned to this managed
     *            connection
     * @throws IllegalStateException
     *             connection is not instanceof <code>CciConnection</code>
     */
    public void associateConnection(Object connection) throws ResourceException {
        final String SIGNATURE = "associateConnection(Object connection)";
        TRACE.entering(SIGNATURE);

        checkIfDestroyed();
        if (connection instanceof CCIConnection) {
            CCIConnection cciCon = (CCIConnection) connection;
            cciCon.associateConnection(this);
        } else {
            IllegalStateException ise = new IllegalStateException("Invalid connection object: " + connection);
            TRACE.throwing(SIGNATURE, ise);
            throw ise;
        }
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Store a new connection event listener (instantiated by the (J2EE server)
     * container) for this managed connection. Connection change events will be
     * send to that listener from now on. (SPI JCA 1.0)
     * 
     * @param listener
     *            <code>ConnectionEventListener</code> to store.
     */
    public void addConnectionEventListener(ConnectionEventListener listener) {
        final String SIGNATURE = "addConnectionEventListener(ConnectionEventListener listener)";
        TRACE.entering(SIGNATURE);
        cciListener.addConnectorListener(listener);
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Remove a connection event listener (instantiated by the (J2EE server)
     * container) from this managed connection. Connection change events won't
     * be send to that listener anymore. (SPI JCA 1.0)
     * 
     * @param listener
     *            <code>ConnectionEventListener</code> to remove.
     */
    public void removeConnectionEventListener(ConnectionEventListener listener) {
        final String SIGNATURE = "removeConnectionEventListener(ConnectionEventListener listener)";
        TRACE.entering(SIGNATURE);
        cciListener.removeConnectorListener(listener);
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Get the two-way JTA transaction if this kind of transaction is supported
     * by the adapter. Since this sample implementation does not supports
     * <code>XAResources</code> it throws always an exception. (SPI JCA 1.0)
     * 
     * @return JTA transaction as <code>XAResource</code> to remove.
     * @throws NotSupportedException
     *             Always thrown
     */
    public XAResource getXAResource() throws ResourceException {
        throw new NotSupportedException("XA transaction not supported");
    }

    /**
     * Get the JCA local transaction if this kind of transaction is supported by
     * the adapter. Since this sample implementation does not supports
     * <code>LocalTransaction</code> it throws always an exception. (SPI JCA
     * 1.0)
     * 
     * @return JTA transaction as <code>XAResource</code> to remove.
     * @throws NotSupportedException
     *             Always thrown
     */
    public javax.resource.spi.LocalTransaction getLocalTransaction() throws ResourceException {
        throw new NotSupportedException("Local transaction not supported");
    }

    /**
     * Returns the descriptive and configuration data for this managed
     * connection as <code>ManagedConnectionMetaData</code>. (SPI JCA 1.0)
     * 
     * @return Managed connection meta data
     * @throws IllegalStateException
     *             Thrown if connection is already deleted
     */
    public ManagedConnectionMetaData getMetaData() throws ResourceException {
        checkIfDestroyed();
        return new SPIManagedConnectionMetaData(this);
    }

    /**
     * Sets the JCA J2EE logwriter <code>PrintWriter</code> object. Although JCA
     * specifies this mechansim it is not being used by XI AF. Instead the
     * resource adapters should use the XI AF trace service classes as done here
     * in this sample.
     * 
     * (SPI JCA 1.0)
     * 
     * @param out
     *            <code>PrintWriter</code> print writer for logging purposes
     */
    public void setLogWriter(PrintWriter out) throws ResourceException {
        final String SIGNATURE = "setLogWriter(PrintWriter out)";
        TRACE.entering(SIGNATURE, new Object[] { out });
        this.logWriter = out;
        out.print("XI AF Sample Adapter has received a J2EE container log writer.");
        out.print("XI AF Sample Adapter will not use the J2EE container log writer. See the trace file for details.");
        TRACE.exiting(SIGNATURE);
    }

    /**
     * Gets the JCA J2EE logwriter <code>PrintWriter</code> object. (SPI JCA
     * 1.0)
     * 
     * @return <code>PrintWriter</code> print writer for logging purposes
     */
    public PrintWriter getLogWriter() throws ResourceException {
        return logWriter;
    }

    /* Now the non SPI methods start */

    /**
     * Returns the "state" of this managed connection (ra implementation
     * specific)
     * 
     * @return true, if this managed connection is already destroyed, else false
     */
    boolean isDestroyed() {
        return destroyed;
    }

    /**
     * Returns the userName/password credential used by this managed connection
     * (ra implementation specific)
     * 
     * @return credential of this managed connection
     */
    PasswordCredential getPasswordCredential() {
        return credential;
    }

    /**
     * Sends a <code>ConnectionEvent</code> to all registered
     * <code>ConnectionEventListeners</code>. (ra implementation specific)
     * 
     * @param eventType
     *            One of the <code>ConnectionEvent</code> event types
     * @param ex
     *            Exception if event is related to an exception, might be
     *            <code>null</code>
     */
    public void sendEvent(int eventType, Exception ex) {
        cciListener.sendEvent(eventType, ex, null);
    }

    /**
     * Sends a <code>ConnectionEvent</code> to all registered
     * <code>ConnectionEventListeners</code>. (ra implementation specific)
     * 
     * @param eventType
     *            One of the <code>ConnectionEvent</code> event types
     * @param ex
     *            Exception if event is related to an exception, might be
     *            <code>null</code>
     * @param connectionHandle
     *            An object that represents the "handle" to the
     *            <code>ManagedConnection</code>, might be <code>null</code>
     */
    public void sendEvent(int eventType, Exception ex, Object connectionHandle) {
        cciListener.sendEvent(eventType, ex, connectionHandle);
    }

    /**
     * Adds a CCI connection in order to operate on this managed connection (ra
     * implementation specific)
     * 
     * @param cciCon
     *            CCI connection to add
     */
    public void addCciConnection(CCIConnection cciCon) {
        connectionSet.add(cciCon);
    }

    /**
     * Removes a CCI connection which operates on this managed connection (ra
     * implementation specific)
     * 
     * @param cciCon
     *            CCI connection which must be removed
     */
    public void removeCciConnection(CCIConnection cciCon) {
        connectionSet.remove(cciCon);
    }

    /**
     * Propagates the start request to the mcf (ra implementation specific)
     */
    public void start() throws ResourceException {
        mcf.startMCF();
    }

    /**
     * Propagates the stop request to the mcf (ra implementation specific)
     */
    public void stop() throws ResourceException {
        mcf.stopMCF();
    }

    /**
     * Checks whether this managed connection is already destroyed. If so, it
     * throws an exception. (ra implementation specific)
     * 
     * @throws IllegalStateException
     *             Thrown if connection is already deleted
     */
    private void checkIfDestroyed() throws ResourceException {
        if (destroyed) {
            throw new javax.resource.spi.IllegalStateException("Managed connection is closed");
        }
    }

}