/*
 * $Id: HttpRequestBodyToParamMap.java 20247 2010-11-18 14:23:04Z dirk.olmes $
 * --------------------------------------------------------------------------------------
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package com.mulesoft.adapter;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConstants;
import org.mule.util.StringUtils;

import java.net.URI;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.net.URLCodec;

public class HttpRequestBodyToParamMap extends AbstractMessageTransformer
{
    public HttpRequestBodyToParamMap()
    {
        registerSourceType(DataTypeFactory.OBJECT);
        setReturnDataType(DataTypeFactory.OBJECT);
    }

    @Override
    public Object transformMessage(MuleMessage message, String outputEncoding) throws TransformerException
    {
        Map<String, Object> paramMap = new HashMap<String, Object>();

        try
        {
            String httpMethod = message.getInboundProperty("http.method");
            String contentType = message.getInboundProperty("Content-Type");
            
            boolean isGet = HttpConstants.METHOD_GET.equalsIgnoreCase(httpMethod);
            boolean isPost = HttpConstants.METHOD_POST.equalsIgnoreCase(httpMethod);
            boolean isUrlEncoded = false;
            if (contentType != null)
            {
                isUrlEncoded = contentType.startsWith("application/x-www-form-urlencoded");
            }

            if (!(isGet || (isPost && isUrlEncoded)))
            {
                throw new Exception("The HTTP method or content type is unsupported!");
            }

            String queryString = null;
            if (isGet)
            {
                URI uri = new URI(message.getPayloadAsString(outputEncoding));
                queryString = uri.getRawQuery();
            }
            else if (isPost)
            {
                queryString = new String(message.getPayloadAsBytes());
            }

            if (StringUtils.isNotBlank(queryString))
            {
                addQueryStringToParameterMap(queryString, paramMap, outputEncoding);
            }
        }
        catch (Exception e)
        {
            throw new TransformerException(this, e);
        }

        return paramMap;
    }

    protected void addQueryStringToParameterMap(String queryString, Map<String, Object> paramMap,
        String outputEncoding) throws Exception
    {
        String[] pairs = queryString.split("&");
        for (String pair : pairs)
        {
            String[] nameValue = pair.split("=");
            if (nameValue.length == 2)
            {
                URLCodec codec = new URLCodec(outputEncoding);
                String key = codec.decode(nameValue[0]);
                String value = codec.decode(nameValue[1]);
                addToParameterMap(paramMap, key, value);
            }
        }
    }

    @SuppressWarnings("unchecked")
    protected void addToParameterMap(Map<String, Object> paramMap, String key, String value)
    {
        Object existingValue = paramMap.get(key);
        if (existingValue != null)
        {
            List<Object> values;
            if (existingValue instanceof List<?>)
            {
                values = (List<Object>) existingValue;
            }
            else
            {
                List list = new LinkedList<Object>();
                list.add(existingValue);
                values = list;
            }

            values.add(value);
            paramMap.put(key, values);
        }
        else
        {
            paramMap.put(key, value);
        }
    }

    @Override
    public boolean isAcceptNull()
    {
        return false;
    }
}
