/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.utils.salesforce;

import com.mulesoft.utils.xml.Element;

import com.sforce.soap.partner.DescribeLayoutResult;
import com.sforce.soap.partner.DescribeSObjectResult;
import com.sforce.soap.partner.Field;
import com.sforce.soap.partner.PartnerConnection;
import com.sforce.soap.partner.SoapType;
import com.sforce.ws.ConnectionException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;


public class TypeGenerator
{
    private SalesforceCallback callback;
    private List<String> typeNames = new ArrayList<String>();
    
    private SortedMap<String, Element> types = new TreeMap<String, Element>();

    public TypeGenerator(SalesforceCallback callback) throws Exception
    {
        this.callback = callback;
        this.typeNames = callback.getTypeNames();
    }

    public void generateType(String name) throws ConnectionException
    {
        if (types.containsKey(name))
        {
            return;
        }
        if (!typeNames.contains(name))
        {
            throw new IllegalArgumentException("The type " + name + " is not known to SalesForce");
        }
        
        generate(name);
    }

    public Map<String, Element> getTypes()
    {
        return new TreeMap<String, Element>(types);
    }

    private void generate(String typeName) throws ConnectionException
    {
        Set<String>  referencedTypes = new HashSet<String>();
        
        DescribeSObjectResult desc = callback.describeSObject(typeName);
        Field[] fields = desc.getFields();
        //dumpFields(fields);
        SortedMap<String, Element> schemaElms = new TreeMap<String, Element>();
        SortedMap<String, Element> fullSchemaElms = new TreeMap<String, Element>();
        
        for (Field field : fields)
        {
            String fName = field.getName();
            if (fName.equals("Id"))
            {
                // Inherited from base type
                continue;
            }

            SoapType fType = field.getSoapType();
            boolean isId = fType == SoapType.ID;
            String fTypeName = getSoapTypeName(fType);
            schemaElms.put(fName, createElement(fName, fTypeName));
            fullSchemaElms.put(fName, createElement(fName, fTypeName));
            if (isId && fName.endsWith("Id"))
            {
                String refTypeName = "Name";
                String[] refTypes = field.getReferenceTo();
                if (refTypes != null && refTypes.length == 1)
                {
                    refTypeName = refTypes[0];
                }
                referencedTypes.add(refTypeName);
                String fullName = fName.substring(0, fName.lastIndexOf("Id"));
                fullSchemaElms.put(fullName, createElement(fullName, "ens:" + fullTypeName(refTypeName)));
            }
        }
        
        createSchemaTypes(typeName, schemaElms.values(), fullSchemaElms.values());

        for (String str : referencedTypes)
        {
            generateType(str);
        }       
    }

    public static String fullTypeName(String refTypeName)
    {
        return refTypeName + "_full";
    }

    private void createSchemaTypes(String typeName, Collection<Element> schemaElms, Collection<Element> fullSchemaElms)
    {
        createSchemaType(typeName, schemaElms);
        createSchemaType(fullTypeName(typeName), fullSchemaElms);
    }

    private void createSchemaType(String typeName, Collection<Element> schemaElms)
    {
        Element top  = new Element("complexType").addAttr("name", typeName);
        Element elm = new Element("complexContent", top);
        elm = new Element("extension", elm).addAttr("base", "ens:sObject");
        elm = new Element("sequence", elm);
        elm.addChild(new Element("element").addAttr("name", "type").addAttr("type", "xsd:string").addAttr("nillable", "false"));
        for (Element schemaElm : schemaElms)
        {
            elm.addChild(schemaElm);
        }
        types.put(typeName, top);
    }

    private Element createElement(String fName, String fTypeName)
    {
        return new Element("element").addAttr("name", fName).addAttr("type", fTypeName).
                        addAttr("nillable", "true").addAttr("minOccurs", "0");
    }

    private String getSoapTypeName(SoapType fType)
    {
        if (fType == SoapType.ID)
        {
            return "tns:ID";
        }
        
        String fTypeStr = fType.toString();
        if (fTypeStr.startsWith("_"))
        {
            fTypeStr = fTypeStr.substring(1);
        }
        return "xsd:" + fTypeStr;
    }

    private void dumpFields(Field[] fields)
    {
        SortedMap<String, Field> sortedFields = new TreeMap<String, Field>();
        
        for (Field field : fields)
        {
            sortedFields.put(field.getName(), field);
        }
        for (Field field : sortedFields.values())
        {
            System.out.println(field.getName() + " " + field.getSoapType() + " " + field.getRelationshipName());;
            if (field.getReferenceTo() != null && field.getReferenceTo().length > 0)
            {
                for (String ref : field.getReferenceTo())
                {
                    System.out.println("\t" + ref);
                }
            }
        }
        System.out.println("------------------");
    }
}
