/* (c) 2011-2012 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.utils.wsdl;

import com.mulesoft.utils.salesforce.TypeGenerator;
import com.mulesoft.utils.schema.ComplexTypes;
import com.mulesoft.utils.schema.SimpleTypes;
import com.mulesoft.utils.xml.Element;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;


public class Wsdl extends WsdlNode
{

    public static final String WSDL_NS = "http://schemas.xmlsoap.org/wsdl/";
    public static final String ENS = "urn:sobject.enterprise.soap.sforce.com";
    public static final String TNS = "urn:enterprise.soap.sforce.com";
    public static final String FNS = "urn:fault.enterprise.soap.sforce.com";
    public static final String SCHEMA_NS = "http://www.w3.org/2001/XMLSchema";
    public static final String SOAP_NS = "http://schemas.xmlsoap.org/wsdl/soap/";

    public static final String PORT_TYPE_NAME = "Soap";
    public static final String BINDING_NAME = "SoapBinding";
    public static final String SERVICE_NAME = "SforceService";
    public static final String PORT_NAME = "Soap";
    public static final String SOAP_ADDRESS = "https://login.salesforce.com/services/Soap/c/23.0";
    
    private Map<String, Element> addedTypes = new TreeMap<String, Element>();
    private Map<String, Element> elements = new TreeMap<String, Element>();
    private List<Operation> operations = new ArrayList<Operation>();
    private List<BindingOperation> bindingOps = new ArrayList<BindingOperation>();
    private Map<String, Message> messages = new TreeMap<String, Message>();
    
    private String[] createFaults = 
            {"InvalidSObjectFault", "UnexpectedErrorFault", "InvalidIdFault", "InvalidFieldFault"};
    private String[] deleteFaults = 
            {"UnexpectedErrorFault"};
    private String[] queryFaults = 
            {"InvalidSObjectFault", "InvalidFieldFault", "MalformedQueryFault", "InvalidIdFault", 
             "UnexpectedErrorFault", "InvalidQueryLocatorFault" };
    private String[] upsertFauls = 
            {"InvalidSObjectFault", "UnexpectedErrorFault", "InvalidIdFault", "InvalidFieldFault" };
    
    public Wsdl()
    {
        super("definitions");
    }

    @Override
    public Element generate()
    {
        Element top = new Element(getName()).addNsDef(WSDL_NS).addNsDef("ens", ENS).addNsDef("tns", TNS)
                        .addNsDef("fns", FNS).addNsDef("xsd", SCHEMA_NS).addNsDef("soap", SOAP_NS)
                        .addAttr("targetNamespace", TNS);
        Element types = new Element("types");
        top.addChild(types);
        types.addChild(generateSchema(TNS));
        types.addChild(generateSchema(ENS));
        types.addChild(generateSchema(FNS));


        for (Message message : messages.values())
        {
            top.addChild(message.generate());
        }

        PortType portType = new PortType(PORT_TYPE_NAME);
        for (Operation op : operations)
        {
            portType.addOperation(op);
        }
        top.addChild(portType.generate());
        
        Binding binding = new Binding(BINDING_NAME, portType);
        for (BindingOperation op : bindingOps)
        {
            binding.addOperation(op);
        }
        top.addChild(binding.generate());

        Port port = new Port(PORT_TYPE_NAME, binding, SOAP_ADDRESS);
        Service service = new Service(SERVICE_NAME);
        service.addPort(port);
        top.addChild(service.generate());

        return top;
    }

    public void addType(String name, Element elm)
    {
        addedTypes.put(name, elm);
    }

    public void addOperation(OperationType type, String typeName)
    {
        String name = type.toString().toLowerCase() + typeName;
        Operation operation = new Operation(name);
        operations.add(operation);
        BindingOperation bindingOp = new BindingOperation(name);
        bindingOps.add(bindingOp);

        String[] faults = new String[0];

        Element requestElement = new Element("element").addAttr("name", name);
        Element responseElement = new Element("element").addAttr("name", name + "Response");
        Element requestSequence = createSequence(requestElement);
        Element responseSequence = createSequence(responseElement);
        switch(type)
        {
            case CREATE:
                faults = createFaults;
                requestSequence.addChild(new Element("element").addAttr("name", "sObjects").addAttr("type", "ens:" + typeName)
                                            .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                responseSequence.addChild(new Element("element").addAttr("name", "result").addAttr("type", "tns:SaveResult")
                        .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                break;
            
            case DELETE:
                faults = deleteFaults;
                requestSequence.addChild(new Element("element").addAttr("name", "ids").addAttr("type", "tns:ID")
                        .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                responseSequence.addChild(new Element("element").addAttr("name", "result").addAttr("type", "tns:DeleteResult")
                        .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                break;
            
            case QUERY:
                String fullTypeName = typeName + "_full";
                faults = queryFaults;
                String queryTypeName = ComplexTypes.getQueryTypeName(fullTypeName);
                addType(queryTypeName, ComplexTypes.getQueryResult("ens", fullTypeName));
                requestSequence.addChild(new Element("element").addAttr("name", "queryString").addAttr("type", "xsd:string"));
                responseSequence.addChild(new Element("element").addAttr("name", "result").addAttr("type", "ens:" + queryTypeName));
                break;
            
            case UPSERT:
                faults = upsertFauls;
                requestSequence.addChild(new Element("element").addAttr("name", "externalIDFieldName").addAttr("type", "xsd:string"));
                requestSequence.addChild(new Element("element").addAttr("name", "sObjects").addAttr("type", "ens:" + typeName)
                        .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                responseSequence.addChild(new Element("element").addAttr("name", "result").addAttr("type", "tns:UpsertResult")
                        .addAttr("minOccurs", "0").addAttr("maxOccurs", "unbounded"));
                break;
            
        }
        messages.put(name + "Request", new Message(name).getRequest());
        messages.put(name + "Response", new Message(name).getResponse());
        elements.put(name + "Request", requestElement);
        elements.put(name + "Response", responseElement);
        String subscribeName = "subscribe" + TypeGenerator.fullTypeName(typeName);
        String subscribeResponseName = subscribeName + "Response";
        elements.put(subscribeResponseName, ComplexTypes.getSubscribeResponse("ens", TypeGenerator.fullTypeName(typeName)));
        Message subscribeResponseMessage = new Message(subscribeName).getResponse();
        messages.put(subscribeResponseName, subscribeResponseMessage);
        addFaults(operation, bindingOp, faults);
    }

    private Element createSequence(Element elm)
    {
        Element sequence = new Element("sequence");
        Element complexType = new Element("complexType");
        elm.addChild(complexType);
        complexType.addChild(sequence);
        return sequence;
    }

    private void addFaults(Operation operation, BindingOperation bindingOp, String[] faults)
    {
        for (String fault : faults)
        {
            operation.addFault(fault);
            bindingOp.addFaultName(fault);
            messages.put(fault, new Message(fault).getFault());
        }
    }

    private Element generateSchema(String ns)
    {
        Schema schema = null;
        if (ns.equals(ENS))
        {
            schema = new Schema("ens", ns, TNS);
        }
        else if (ns.equals(TNS))
        {
            schema = new Schema("tns", ns, ENS);
        }
        else if (ns.equals(FNS))
        {
            schema = new Schema("fns", ns);
        }
        else
        {
            schema = new Schema("ns", ns);    
        }
        
        Element elm = schema.generate();
        elm.addChildren(SimpleTypes.getSimpleTypes(ns));
        elm.addChildren(ComplexTypes.getComplexTypes(ns));
        if (ns.equals(ENS))
        {
            elm.addChildren(new ArrayList(addedTypes.values()));
        }
        else if (ns.equals(TNS))
        {
            elm.addChildren(new ArrayList(elements.values()));
        }
        return elm;
    }
}
