package com.sap.aii.af.sample.adapter.ra;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;

import com.sap.aii.af.service.administration.api.cpa.CPAFactory;
import com.sap.aii.af.service.administration.api.cpa.CPALookupManager;
import com.sap.aii.af.service.cpa.AbstractPartyCallBackHandler;
import com.sap.aii.af.service.cpa.CPAException;
import com.sap.aii.af.service.cpa.Party;
import com.sap.aii.af.service.cpa.PartyIdentifier;

/**
 * PartyChangeCallBackHandler handles the PartyChangeCallBack Events.
 * This Handler is implemented as a Singleton for Demonstration purposes and 
 * for the sake of simplicity.
 * 
 * The Real Implementations need not be a Singleton, and Multiple Instances of the
 * Callback Handler can be registered for PartyChangeEvent of one Party and Each instance 
 * can be registered for any number of parties.
 * 
 * Also the registration should be undone at the application stop or whenever
 * it is not needed by calling the unregister Method on the PartyCallBackController
 * 
 * This implementation checks the identifier for the schema "DUNS" in the 
 * party configuration.
 * 
 * @author i044316
 */
public class PartyChangeCallBackHandler extends AbstractPartyCallBackHandler{

	private static final XITrace TRACE = new XITrace(PartyChangeCallBackHandler.class.getName());
	public static final String DUNS = "DUNS" ;
	private static PartyChangeCallBackHandler partyChangeCallBackHandler = new PartyChangeCallBackHandler () ;
	//Map that holds PartyName and the PartyIdentifier
	private  HashMap<String, String> partyidentifierMap = null ;
	
	private PartyChangeCallBackHandler (){
		partyidentifierMap = new HashMap<String, String> () ;
	}
	
	public static PartyChangeCallBackHandler getInstance () {
		return partyChangeCallBackHandler ;
	}
	
	@Override
	/**
	 * This is the callback method which is being called from the CPA service whenever
	 * the Identifier of the Party changes.
	 * 
	 * Since there is a possiblity that this method can be called during the delete event,
	 * a check is made with the old Party identifier with the new PartyIdentifier and then the
	 * change is updated locally
	 * 
	 * Also since there is a possibility that there can be multiple identifiers for a party
	 * a check for the PartySchema is done before making an update.
	 */
	public void partyrefreshEvent(Party arg0) {
		final String SIGNATURE = "partyrefreshEvent(Party arg0)";
		TRACE.entering(SIGNATURE, new Object[] {arg0});
		try{
			String partyName = arg0.getParty() ;
			synchronized ( partyidentifierMap )
			{
				if ( partyidentifierMap.containsKey( partyName ) )
				{
					CPALookupManager lookupManager = CPAFactory.getInstance().getLookupManager() ;
					LinkedList<PartyIdentifier> list = lookupManager.getPartyIdentifiersByParty(arg0)  ;
					Iterator<PartyIdentifier> it = list.iterator() ;
					String oldId = partyidentifierMap.get( partyName ) ;
					while ( it.hasNext() )
					{
						PartyIdentifier partyIdentifier = it.next() ;
						String id = partyIdentifier.getPartyIdentifier() ;
						String schema = partyIdentifier.getPartySchema() ;
						if ( schema.equals( DUNS ) && ! id.equals( oldId ) ){
							TRACE.infoT(SIGNATURE, "Received PartyChangeEvent for the Party " + arg0.getParty() 
									+ " The new Identifier is " + id ) ;
							partyidentifierMap.put( partyName , id ) ;
						}
					}
				}
			}
		}catch ( CPAException e ){
			TRACE.catching ( SIGNATURE, e ) ;
		}
		TRACE.exiting(SIGNATURE);
	}
	
	public void addParty (String partyName, String identifier ){
		partyidentifierMap.put( partyName, identifier ) ;
	}
	
	public void removeParty ( String partyName ){
		partyidentifierMap.remove( partyName ) ;
	}

	public Set<String> getRegisteredParties () {
		return partyidentifierMap.keySet() ;
	}
	
	/**
	 * Clears the Registrations in the Map
	 */
	public void clear() {
		partyidentifierMap.clear() ;
	}
}
