/*
 * (C) 2006 SAP XI 7.1 Adapter Framework Module Sample
 */

package com.sap.aii.af.sample.module;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

import com.sap.engine.interfaces.messaging.api.Message;
import com.sap.engine.interfaces.messaging.api.Payload;
import com.sap.engine.interfaces.messaging.api.TextPayload;
import com.sap.engine.interfaces.messaging.api.XMLPayload;

import com.sap.aii.af.lib.mp.module.Module;
import com.sap.aii.af.lib.mp.module.ModuleContext;
import com.sap.aii.af.lib.mp.module.ModuleData;
import com.sap.aii.af.lib.mp.module.ModuleException;
import com.sap.aii.af.service.administration.api.cpa.CPAFactory;
import com.sap.aii.af.service.administration.api.cpa.CPALookupManager;
import com.sap.aii.af.service.cpa.Channel;
import com.sap.aii.af.service.cpa.CPAObject;
import com.sap.aii.af.service.cpa.CPAObjectType;
import com.sap.aii.af.service.resource.SAPSecurityResources;

import com.sap.security.api.ssf.ISsfProfile;

/**
 * <code>ConvertCRLFfromToLF</code> represents a XI Adapter Framework (AF) compliant
 * module that can be called by the XI AF Module Processor (MP). The XI AF MP concept
 * allows to implement customer exits as well as extensions for arbitrary adapters.
 * A conceptual description can be found in the XI AF API documentation, see
 * the NetWeaver Online help.  
 * In general a AF module is a pojo module.
 * This particualr module can be simply used with module name
 * com.sap.aii.adapter.sample.module.ConvertCRLFfromToLF0
 * and module type java class.
 * @version: $Id: //tc/xpi.external/NW730EXT_01_REL/src/_sample_rar_module/rar/src/com/sap/aii/af/sample/module/ConvertCRLFfromToLF0.java#1 $
 **/
public class ConvertCRLFfromToLF0 implements Module {

	static final long serialVersionUID = 7435850550539048631L;   //See Serializable IF
	
	private static final String LINE_SEP   = System.getProperty("line.separator");
    private static final String CRLF       = "\r\n";
    private static final String LF         = "\n";


	/** 
	 * The main method of AF modules is the <code>process()</code> method. It takes the XI message, changes it
	 * according to some module specific rules and forwards it in the module chain. If this module is the last
	 * module in the chain before the adapter is being called it must ensure that in case of synchronous messages
	 * a response message is sent back in the return <code>ModuleDate</code> parameter.
	 * This sample here converts carriage return line feeds (CRLF) to line feeds (LF) and vice versa for all
	 * text or XML attachments of the XI message.
	 * @param moduleContext Contains data of the module processor that might be important for the module implementation 
	 * such as current channel ID
	 * @param inputModuleData Contains the input XI message as principal data plus eventual set supplemental data
	 * @return ModuleData Contains the (changed) output XI message. Might be the response message if the module
	 * is the last in the chain.
	 * @exception ModuleException Describes the cause of the exception and indicates whether an retry is sensible or not. 
	 */
	public ModuleData process(ModuleContext moduleContext, ModuleData inputModuleData) throws ModuleException {
		//First initialize the trace and log the module entry
		String SIGNATURE = "process(ModuleContext moduleContext, ModuleData inputModuleData)";
		Location location = null;
	    
		// Create the location always new to avoid serialization/transient of location
		try {
	       location = Location.getLocation(ConvertCRLFfromToLF0.class.getName());
	    } catch (Exception t) {
	       t.printStackTrace();
	       ModuleException me = new ModuleException("Unable to create trace location", t);
	       throw me;
	    }
		
	    location.entering(SIGNATURE, new Object[] { moduleContext, inputModuleData });

		// Access the XI message. I.e. this module must be placed in the module chain
		// behind a module that sets an XI message as principal data.
		Object obj = null;
		Message msg = null;
		try {
			obj = inputModuleData.getPrincipalData();
			msg = (Message) obj;
		} catch (Exception e) {
			locationCatching(SIGNATURE, e, location);
			ModuleException me = new ModuleException(e);
			location.throwing(SIGNATURE, me);
			throw me;
		}
		
		if (msg == null) {
			String errTxt = "Null as XI message received (PrincipalData in ModulData is null)";
			location.errorT(SIGNATURE, errTxt);
			ModuleException me = new ModuleException(errTxt);
			location.throwing(SIGNATURE, me);
			throw me;
		}

		// Read the channel ID, channel and the module configuration
		String cid  = null;
		String mode = null;
		Channel channel = null;
		try {
			// CS_GETMODDAT START
			mode = (String) moduleContext.getContextData("mode");
			// CS_GETMODDAT END
			// CS_GETCHADAT START
			cid  = moduleContext.getChannelID();
			// Access channel configuration only if sure that this module runs only in this channel-adapter type
			CPALookupManager lm = CPAFactory.getInstance().getLookupManager();
			channel = (Channel) lm.getCPAObject(CPAObjectType.CHANNEL, cid);
			// Example to access a channel configuration parameter in a module: String someParameter = channel.getValueAsString("YourAttributeName");
			// CS_GETCHADAT END
			if (mode == null) {
				location.debugT(SIGNATURE, "Mode parameter is not set. Switch to 'none' as default.");
				mode = "none";
			}
			location.debugT(SIGNATURE, "Mode is set to {0}", new Object[] {mode});
		} catch (Exception e) {
			locationCatching(SIGNATURE, e, location);
			location.errorT(SIGNATURE, "Cannot read the module context and configuration data");
			ModuleException me = new ModuleException(e);
			location.throwing(SIGNATURE, me);
			throw me;
		}

		// Bypass the conversion for test purposes
		if (mode.compareToIgnoreCase("none") == 0)  { 
			location.debugT(SIGNATURE, "Bypass CRLF conversion since 'mode' parameter was set to 'none'.");
		}
        // Bypass the conversion for CRLFtoNative mode if CRLF is the native line ending
        else if (mode.compareToIgnoreCase("CRLFtoNative") == 0 && CRLF.equals(LINE_SEP)) {
            location.debugT(SIGNATURE, "Bypass CRLF conversion since 'mode' parameter was set to 'CRLFtoNative' and the native line separator is CRLF.");
        } 
        // Bypass the conversion for LFtoNative mode if LF is the native line ending
        else if (mode.compareToIgnoreCase("LFtoNative") == 0 && LF.equals(LINE_SEP)) {
            location.debugT(SIGNATURE, "Bypass CRLF conversion since 'mode' parameter was set to 'LFtoNative' and the native line separator is LF.");
        } 
		else {
			// Convert the payloads of the message
			// Please note: the various message getters return references to the payloads
			// I.e. the input message itself is changed here.
			try {
				// Convert the main XML document
				XMLPayload xmlpayload = msg.getDocument();
				if (xmlpayload != null) {
					// Mode CRLF to LF
					if (mode.compareToIgnoreCase("CRLFtoLF") == 0)  {
						xmlpayload.setContent(convertCRLFtoLF(xmlpayload.getContent(), location));			
					}
                    // Mode CRLF to Native
                    else if (mode.compareToIgnoreCase("CRLFtoNative") == 0) {
                        xmlpayload.setContent(convertCRLFtoNative(xmlpayload.getContent(), location));            
                    }
                    // Mode LF to Native
                    else if (mode.compareToIgnoreCase("LFtoNative") == 0) {
                        xmlpayload.setContent(convertLFtoNative(xmlpayload.getContent(), location));            
                    }
					// Mode LF to CRLF
					else {
						xmlpayload.setContent(convertLFtoCRLF(xmlpayload.getContent(), location));			
					}
				}
				// Convert the payloads
				java.util.Iterator iter = msg.getAttachmentIterator();
				Payload payload = null;
				while (iter.hasNext()) {
					payload = (Payload) iter.next();	
					if (payload instanceof TextPayload) {
						TextPayload textpayload = (TextPayload) payload;
						// Mode CRLF to LF
						if (mode.compareToIgnoreCase("CRLFtoLF") == 0)  {
							textpayload.setContent(convertCRLFtoLF(textpayload.getContent(), location));			
						}
                        // Mode CRLF to Native
                        else if (mode.compareToIgnoreCase("CRLFtoNative") == 0) {
                            textpayload.setContent(convertCRLFtoNative(textpayload.getContent(), location));            
                        }
                        // Mode LF to Native
                        else if (mode.compareToIgnoreCase("LFtoNative") == 0) {
                            textpayload.setContent(convertLFtoNative(textpayload.getContent(), location));            
                        }
						// Mode LF to CRLF
						else {
							textpayload.setContent(convertLFtoCRLF(textpayload.getContent(), location));			
						}
					}		
				}
				
				// Keep the message as principle data since the message itself was changed
				inputModuleData.setPrincipalData(msg);
				location.debugT(SIGNATURE, "CRLF conversion finished sucessfully.");
			} catch (Exception e) {
				locationCatching(SIGNATURE, e, location);
				location.errorT(SIGNATURE, "Cannot convert one of the payloads. Reason: {0}", new Object[] {e.getMessage()});
				ModuleException me = new ModuleException(e);
				location.throwing(SIGNATURE, me);
				throw me;
			}
		}

		location.exiting(SIGNATURE);
		return inputModuleData;
	}

	/**
	 * Converts LF in text payloads to CRLF 
	 * @param src Byte array with text data (UTF-8 is assumed)
	 * @param location trace location, created per bean method call
	 * @return Converted byte array
	 */
	private byte[] convertLFtoCRLF(byte[] src, Location location) {
		String SIGNATURE = "convertLFtoCRLF(byte[] src)";
		location.entering(SIGNATURE, new Object[] { src });
		byte[] buf = new byte[2 * src.length];
		// theoret. maximum: LF to CRLF for all bytes doubles chunk size
		int actualCount = 0;
		int maxCount = 0;
		for (int i = 0; i < src.length; i++) {
			if (src[i] == '\n') {
				buf[actualCount] = '\r';
				buf[actualCount + 1] = '\n';
				actualCount += 2;
			} else {
				buf[actualCount++] = src[i];
			}
		}
		byte[] dst = new byte[actualCount];
		System.arraycopy(buf,0,dst,0,actualCount);
		location.debugT(SIGNATURE, "Found {0} LFs that were replaced by CRLF", new Object[] {String.valueOf(actualCount-src.length)});
		location.exiting(SIGNATURE);
		return dst;
	}

	/**
	 * Converts CRLF in text payload to LF 
	 * @param src Byte array with text data (UTF-8 is assumed)
	 * @param location trace location, created per bean method call
	 * @return Converted byte array 
	 */
	private byte[] convertCRLFtoLF(byte[] src, Location location) {
		String SIGNATURE = "convertCRLFtoLF(byte[] src)";
		location.entering(SIGNATURE, new Object[] { src });
		int i;
		int dstlen;
		int srclen = src.length;
		dstlen = 0;
		for (i = 0; i < srclen; i++) {
			if (src[i] != '\n') { // from \r\n to \n: ignore \n, replace \r
				if (src[i] != '\r') {
					src[dstlen++] = src[i];
				} else {
					src[dstlen++] = '\n';
				}
			}
		}
		byte[] dst = new byte[dstlen];
		System.arraycopy(src,0,dst,0,dstlen);
		location.debugT(SIGNATURE, "Found {0} CRLFs that were replaced by LF", new Object[] {String.valueOf(srclen - dstlen)});
		location.exiting(SIGNATURE);
		return dst;
	}

    /**
     * Converts LF in text payloads to the native line ending
     * @param src Byte array with text data (UTF-8 is assumed)
	 * @param location trace location, created per bean method call
     * @return Converted byte array
     */
    private byte[] convertLFtoNative(byte[] src, Location location) {
        String SIGNATURE = "convertLFtoNative(byte[] src)";
        location.entering(SIGNATURE, new Object[] { src });
        byte[] buf = new byte[2 * src.length];
        // theoret. maximum: LF to CRLF for all bytes doubles chunk size
        int actualCount = 0;
        int maxCount = 0;
        int replacedCount = 0;
        for (int i = 0; i < src.length; i++) {
            if (src[i] == '\n') {
                for (int j = 0; j < LINE_SEP.length(); j++)
                    src[actualCount++] = (byte) LINE_SEP.charAt(j);
                replacedCount++;
            } else {
                buf[actualCount++] = src[i];
            }
        }
        byte[] dst = new byte[actualCount];
        System.arraycopy(buf,0,dst,0,actualCount);
        location.debugT(SIGNATURE, "Found {0} LFs that were replaced by the native line ending", new Object[] {String.valueOf(replacedCount)});
        location.exiting(SIGNATURE);
        return dst;
    }

    /**
     * Converts CRLF in text payload to the native line ending 
     * @param src Byte array with text data (UTF-8 is assumed)
	 * @param location trace location, created per bean method call
     * @return Converted byte array 
     */
    private byte[] convertCRLFtoNative(byte[] src, Location location) {
        String SIGNATURE = "convertCRLFtoNative(byte[] src)";
        location.entering(SIGNATURE, new Object[] { src });
        int i;
        int dstlen = 0;
        int srclen = src.length;
        int replacedCount = 0;
        for (i = 0; i < srclen; i++) {
            if (i + 1 < srclen && src[i] == '\r' && src[i + 1] == '\n') {
                for (int j = 0; j < LINE_SEP.length(); j++)
                    src[dstlen++] = (byte) LINE_SEP.charAt(j);
                i++;
                replacedCount++;
            } else
                src[dstlen++] = src[i];
        }
        byte[] dst = new byte[dstlen];
        System.arraycopy(src,0,dst,0,dstlen);
        location.debugT(SIGNATURE, "Found {0} CRLFs that were replaced by the native line ending", new Object[] {String.valueOf(replacedCount)});
        location.exiting(SIGNATURE);
        return dst;
    }
    /**
     * Helper method that writes a location entry that the specified throwable was caught.
     * @param  signature   Signature of the method
	 * @param location trace location, created per bean method call
     * @param  t     Throwable
     */
    private void locationCatching(String signature, Throwable t, Location location) {
      if (location != null) {
        if(location.beLogged(Severity.WARNING)){
            // only build String from Stacklocation if it is needed...   
  	      ByteArrayOutputStream oStream = new ByteArrayOutputStream(1024);
  	      PrintStream pStream = new PrintStream(oStream);
  	      t.printStackTrace(pStream);
  	      pStream.close();
  	      String stackTrace = oStream.toString();
  	      location.warningT(signature, "Catching {0}", new Object[] {stackTrace});
        }
      }
    }
}
