/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.data.sample;

import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.annotation.param.Config;
import org.mule.sdk.api.annotation.param.Connection;
import org.mule.sdk.api.annotation.param.Parameter;
import org.mule.sdk.api.runtime.operation.Result;
import org.mule.sdk.api.runtime.source.Source;
import org.mule.sdk.api.runtime.streaming.PagingProvider;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.List;

/**
 * Implement this interface to enhance operations and {@link Source message sources} with sample data that is representative
 * of the one that the component will produce in real production scenarios. The sample data is represented as a traditional
 * {@link Result} object which contains both the payload and attributes output values. For operation methods that return a
 * single value instead of a {@link Result}, implementations of this interface should return {@code Result<T, Void>}.
 * <p>
 * Implementations can use annotations such as {@link Parameter}, {@link Config} and {@link Connection} to access
 * actual parameters, configs and connections that are required in order to generate the sample data.
 * <p>
 * Implementations also <b>MUST</b> guarantee that the act of fetching that sample data <b>DOES NOT</b> generate any significant
 * side effects on the target system.
 * <p>
 * For connectors, implementations are most valuable when they return real data from the system they connect to. This is the
 * best practice and should be preferred over the option of simply returning hard coded values.
 * <p>
 * This interface defines two generic types for the component's output ({@code T} for the output payload and {@code A} for the
 * output attributes). These must match the actual output types of the enhanced components. For components which returns dynamic
 * types, it's each implementation's responsibility to return instances of the correct dynamic type.
 * <p>
 * <b>Streaming is not supported.</b> When providing sample data for components which return an {@link InputStream},
 * implementations should use a {@link ByteArrayInputStream} or any other &quot;offline&quot; implementation which has
 * already procured the entire data stream before the {@link #getSample()} method returns. The {@link InputStream} <b>MUST</b>
 * be fully consumable even after the underlying connection has already been closed. Using temporal buffer files here is also
 * discouraged.
 * <p>
 * <b>Paging is streaming.</b> The Mule Runtime considers paged operations as a particular case of streaming. Therefore, same
 * limitations of {@link InputStream} applies. For operations which return a {@link PagingProvider}, a {@link List} of the
 * same item types should be used instead. For example, for this method:
 *
 * <pre>
 *   public PagingProvider<Map<String, Object>, Connection> pagedOperation() {
 *
 *   }
 * </pre>
 *
 * The {@link #getSample()} method should be defined like this:
 *
 * <pre>
 *   public Result<List<Map<String, Object>, Void> getSample() {
 *
 *   }
 * </pre>
 * <p>
 * It is a best practice for implementations to not return large objects or data streams. Make sure that the data is
 * relevant yet small enough to fit into the memory of a small worker instance.
 * <p>
 * Implementations <b>MUST</b> provide a default constructor.
 *
 * @param <T> the generic type of the output value
 * @param <A> the generic type of the message attributes
 * @since 1.4
 */
@MinMuleVersion("4.4")
public interface SampleDataProvider<T, A> {

  /**
   * An id which identifies each implementation of this interface. It <b>MUST</b> be unique among implementations in the same
   * extension. Subsequent invocations <b>MUST</b> always return the same fixed value.
   *
   * @return the resolver's id
   */
  String getId();

  /**
   * Obtains and returns the sample information.
   *
   * @return a {@link Result} object
   * @throws SampleDataException if the sample cannot be obtained
   */
  Result<T, A> getSample() throws SampleDataException;
}
