/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.client;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.domain.message.request.HttpRequest;
import org.mule.sdk.api.http.domain.message.response.HttpResponse;
import org.mule.sdk.api.http.sse.client.SseSource;
import org.mule.sdk.api.http.sse.client.SseSourceConfigurer;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * Object to send HTTP requests. Notice it must be started to be used and stopped to be disposed properly.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpClient extends Startable, Stoppable {

  /**
   * Sends an HttpRequest without blocking the current thread. When a response is available or the request times out, the returned
   * {@link CompletableFuture} will be completed. Be aware that the response body processing will be deferred so that the response
   * can be processed even when a large body is still being received. If the full response is needed right away then the provided
   * {@link HttpResponse} must be read in a different thread so that it does not block the {@link HttpClient} threads handling the
   * response. It's therefore any of the async methods available, such as
   * {@link CompletableFuture#whenCompleteAsync(BiConsumer, Executor)}, to handle the response is those scenarios since they
   * guarantee executing on a different thread.
   *
   * @param request the {@link HttpRequest} to send
   * @param options a callback to configure the request options.
   * @return a {@link CompletableFuture} that will complete once the {@link HttpResponse} is available
   */
  CompletableFuture<HttpResponse> sendAsync(HttpRequest request, Consumer<HttpRequestOptionsConfigurer> options);

  /**
   * Creates a consumer of Server-sent events. The resulting {@link SseSource} is not connected automatically.
   *
   * @param configConsumer a callback to configure the resulting source.
   * @return a non-connected instance of {@link SseSource}.
   */
  SseSource sseSource(Consumer<SseSourceConfigurer> configConsumer);
}
