/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.artifact.lifecycle;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.sdk.api.annotation.MinMuleVersion;

import java.util.stream.Stream;

/**
 * Provides context information about the disposal of the Artifact enclosing the Extension.
 *
 * <b>NOTE:</b> Experimental feature. Backwards compatibility not guaranteed.
 * <p>
 *
 * @see ArtifactLifecycleListener
 * @since 1.0
 */
@NoImplement
@MinMuleVersion("4.5.0")
@Experimental
public interface ArtifactDisposalContext {

  /**
   * @return the {@link ClassLoader} for the extension that defines the {@link ArtifactLifecycleListener} being executed
   */
  ClassLoader getExtensionClassLoader();

  /**
   * @return the {@link ClassLoader} for the enclosing artifact that's being disposed, most typically an app or domain
   */
  ClassLoader getArtifactClassLoader();

  /**
   * @param classLoader a classloader
   * @return whether the given {@code classLoader} is owned by the extension that defines the {@link ArtifactLifecycleListener}
   *         being executed.
   */
  boolean isExtensionOwnedClassLoader(ClassLoader classLoader);

  /**
   * @param classLoader a classloader
   * @return whether the given {@code classLoader} is owned by the enclosing artifact that is being disposed, most typically an
   *         app or domain
   */
  boolean isArtifactOwnedClassLoader(ClassLoader classLoader);

  /**
   * Returns a {@link Stream} with the {@link Thread threads} that were <b>CREATED</b> by the extension.
   * <p>
   * By "created" we refer to Threads that were actually explicitly instantiated by this extension or one of its libraries.
   * <p>
   * This <b>DOES NOT</b> include threads that served tasks submitted through the {@link SchedulerService}, since those threads
   * actually belong to the Mule Runtime. For those, you still need to dispose the {@link Scheduler schedulers} as specified in
   * their API.
   *
   * @return a {@link Stream} of {@link Thread threads}
   * @see #isExtensionOwnedThread(Thread)
   */
  Stream<Thread> getExtensionOwnedThreads();

  /**
   * Returns a {@link Stream} with the {@link Thread threads} that were <b>CREATED</b> by the enclosing artifact that is being
   * disposed.
   * <p>
   * By "created" we refer to Threads that were actually explicitly instantiated by the enclosing artifact or one of its
   * libraries.
   * <p>
   * This <b>DOES NOT</b> include threads that served tasks submitted through the {@link SchedulerService}, since those threads
   * actually belong to the Mule Runtime. For those, you still need to dispose the {@link Scheduler schedulers} as specified in
   * their API.
   *
   * @return a {@link Stream} of {@link Thread threads}
   * @see #isArtifactOwnedThread(Thread)
   */
  Stream<Thread> getArtifactOwnedThreads();

  /**
   * @param thread a {@link Thread}
   * @return whether the given {@code thread} is owned by the enclosing artifact. This will follow the same criteria described in
   *         {@link #getArtifactOwnedThreads()}.
   * @see #getArtifactOwnedThreads()
   */
  boolean isArtifactOwnedThread(Thread thread);

  /**
   * @param thread a {@link Thread}
   * @return whether the given {@code thread} is owned by either the extension that defines the {@link ArtifactLifecycleListener}
   *         being executed. This will follow the same criteria described in {@link #getExtensionOwnedThreads()}.
   * @see #getExtensionOwnedThreads()
   */
  boolean isExtensionOwnedThread(Thread thread);
}
