/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.domain.message;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.util.MultiMap;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.domain.entity.HttpEntity;

import java.util.Collection;
import java.util.Optional;

/**
 * Base implementation of an {@link HttpMessage} builder. Implementations should extend it and indicate their type and the type of
 * the message they build.
 *
 * @param <Builder> the type of the builder itself.
 * @param <Message> the type of {@link HttpMessage} that the builder creates.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpMessageBuilder<Builder extends HttpMessageBuilder, Message extends HttpMessage> {

  /**
   * @param entity the {@link HttpEntity} that should be used as body for the {@link HttpMessage}. Non null.
   * @return this builder
   */
  Builder entity(HttpEntity entity);

  /**
   * @param headersMap a {@link MultiMap} representing the HTTP headers of the {@link HttpMessage} desired. Non-null.
   * @return this builder
   */
  Builder headers(MultiMap<String, String> headersMap);

  /**
   * Includes a new header to be sent in the desired {@link HttpMessage}. Since HTTP headers are case insensitive and can have
   * several values, multiple calls to this method using the same header name will accumulate the values and all of them will be
   * sent.
   *
   * @param name  the name of the HTTP header
   * @param value the value of the HTTP header
   * @return this builder
   */
  Builder addHeader(String name, String value);

  /**
   * Includes a new header with multiple values to be sent in the desired {@link HttpMessage}.
   *
   * @param name   the name of the HTTP header
   * @param values the values of the HTTP header
   * @return this builder
   */
  Builder addHeaders(String name, Collection<String> values);

  /**
   * Removes a header. Since multiple values might be present, all are removed.
   *
   * @param name the name of the HTTP header to remove
   * @return this builder
   */
  Builder removeHeader(String name);

  /**
   * Returns the value of a given HTTP request. If there are several then the first will be return. Use
   * {@link #getHeaderValues(String)} instead if all values are required.
   *
   * @param name the name of the desired HTTP header
   * @return the value of the header or {@code null} if there isn't one
   */
  Optional<String> getHeaderValue(String name);

  /**
   * Gives access to all current values of a given HTTP header.
   *
   * @param name the name of the desired HTTP header
   * @return an immutable {@link Collection} of {@link String} values for the header
   */
  Collection<String> getHeaderValues(String name);

  /**
   * @return an immutable version of the current headers in the builder.
   */
  MultiMap<String, String> getHeaders();

  /**
   * Discard this builder after calling this method.
   */
  Message build();

}
