/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.protocol;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.sdk.api.annotation.MinMuleVersion;

/**
 * Allows enabling/disabling and configuring HTTP/2 specific parameters.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface Http2ProtocolConfigurer {

  /**
   * Enables or disables HTTP/2 support.
   *
   * @param enabled whether it should be enabled or not. Default is {@code false}.
   * @return this configurer.
   */
  Http2ProtocolConfigurer setEnabled(boolean enabled);

  /**
   * Allows the sender to inform the remote endpoint of the maximum size of the header compression table used to decode header
   * blocks, in octets. The encoder can select any size equal to or less than this value by using signaling specific to the header
   * compression format inside a header block (see [COMPRESSION]). The initial value is 4,096 octets.
   *
   * @param headerTableSize the value to set.
   * @return this configurer.
   * @see <a href="https://datatracker.ietf.org/doc/html/rfc7540#section-6.5.2">SETTINGS_HEADER_TABLE_SIZE (0x1)</a>
   */
  Http2ProtocolConfigurer setHeaderTableSize(Long headerTableSize);

  /**
   * Indicates the maximum number of concurrent streams that the sender will allow. This limit is directional: it applies to the
   * number of streams that the sender permits the receiver to create. Initially, there is no limit to this value. It is
   * recommended that this value be no smaller than 100, so as to not unnecessarily limit parallelism.
   *
   * A value of 0 for SETTINGS_MAX_CONCURRENT_STREAMS SHOULD NOT be treated as special by endpoints. A zero value does prevent the
   * creation of new streams; however, this can also happen for any limit that is exhausted with active streams. Servers SHOULD
   * only set a zero value for short durations; if a server does not wish to accept requests, closing the connection is more
   * appropriate.
   *
   * @param maxConcurrentStreams the value to set.
   * @return this configurer.
   * @see <a href="https://datatracker.ietf.org/doc/html/rfc7540#section-6.5.2">SETTINGS_MAX_CONCURRENT_STREAMS (0x3)</a>
   */
  Http2ProtocolConfigurer setMaxConcurrentStreams(Long maxConcurrentStreams);

  /**
   * Indicates the sender's initial window size (in octets) for stream-level flow control. The initial value is 2^16-1 (65,535)
   * octets.
   *
   * This setting affects the window size of all streams (see Section 6.9.2).
   *
   * Values above the maximum flow-control window size of 2^31-1 MUST be treated as a connection error (Section 5.4.1) of type
   * FLOW_CONTROL_ERROR.
   *
   * @param initialWindowSize the value to set.
   * @return this configurer.
   * @see <a href="https://datatracker.ietf.org/doc/html/rfc7540#section-6.5.2">SETTINGS_INITIAL_WINDOW_SIZE (0x4)</a>
   */
  Http2ProtocolConfigurer setInitialWindowSize(Integer initialWindowSize);

  /**
   * Indicates the size of the largest frame payload that the sender is willing to receive, in octets.
   *
   * The initial value is 2^14 (16,384) octets. The value advertised by an endpoint MUST be between this initial value and the
   * maximum allowed frame size (2^24-1 or 16,777,215 octets), inclusive. Values outside this range MUST be treated as a
   * connection error (Section 5.4.1) of type PROTOCOL_ERROR.
   *
   * @param maxFrameSize the value to set.
   * @return this configurer.
   * @see <a href="https://datatracker.ietf.org/doc/html/rfc7540#section-6.5.2">SETTINGS_MAX_FRAME_SIZE (0x5)</a>
   */
  Http2ProtocolConfigurer setMaxFrameSize(Integer maxFrameSize);

  /**
   * This advisory setting informs a peer of the maximum size of header list that the sender is prepared to accept, in octets. The
   * value is based on the uncompressed size of header fields, including the length of the name and value in octets plus an
   * overhead of 32 octets for each header field.
   *
   * For any given request, a lower limit than what is advertised MAY be enforced. The initial value of this setting is unlimited.
   *
   * @param maxHeaderListSize the value to set.
   * @return this configurer.
   * @see <a href="https://datatracker.ietf.org/doc/html/rfc7540#section-6.5.2">SETTINGS_MAX_HEADER_LIST_SIZE (0x6)</a>
   */
  Http2ProtocolConfigurer setMaxHeaderListSize(Long maxHeaderListSize);
}
