/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.sse.client;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.client.HttpRequestOptionsConfigurer;
import org.mule.sdk.api.http.domain.message.request.HttpRequestBuilder;
import org.mule.sdk.api.http.domain.message.response.HttpResponse;

import java.util.function.Consumer;

/**
 * Allows to configure parameters to the {@link SseSource}.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface SseSourceConfigurer {

  /**
   * Useful when you already have an {@link HttpResponse} with an SSE stream. The resulting SSE Source won't do automatic retries.
   *
   * @param response a response that has a {@code text/event-stream} as payload.
   * @return this configurer.
   */
  SseSourceConfigurer withResponse(HttpResponse response);

  /**
   * Sets the URL of the server to send the SSE initiator request to.
   *
   * @param url the server URL.
   * @return this configurer.
   */
  SseSourceConfigurer withUrl(String url);

  /**
   * Configures whether the resulting source allows the server to override the delay for retries.
   *
   * @param allow whether the source allows the override or not.
   * @return this configurer.
   */
  SseSourceConfigurer withAllowRetryDelayOverride(boolean allow);

  /**
   * Configures the initial retry delay of the resulting source.
   *
   * @param initialRetryDelayMillis the delay in milliseconds.
   * @return this configurer.
   */
  SseSourceConfigurer withInitialRetryDelayMillis(long initialRetryDelayMillis);

  /**
   * Configures whether the source will retry when the SSE stream ends.
   *
   * @param shouldRetryOnStreamEnd whether retry or not.
   * @return this configurer.
   */
  SseSourceConfigurer withRetry(boolean shouldRetryOnStreamEnd);

  /**
   * Configures a callback to customize the initiator request builder.
   *
   * @param requestCustomizer the callback.
   * @return this configurer.
   */
  SseSourceConfigurer withRequestCustomizer(Consumer<HttpRequestBuilder> requestCustomizer);

  /**
   * Configures a callback to customize the initiator request options.
   *
   * @param requestOptions the callback.
   * @return this configurer.
   */
  SseSourceConfigurer withRequestOptions(Consumer<HttpRequestOptionsConfigurer> requestOptions);

  /**
   * Configures whether the initiator request has to preserve the header case or not.
   *
   * @param preserveHeadersCase whether the initiator request has to preserve the header case or not.
   * @return this configurer.
   */
  SseSourceConfigurer withPreserveHeadersCase(boolean preserveHeadersCase);
}
