/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.server;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.protocol.Http1ProtocolConfigurer;
import org.mule.sdk.api.http.protocol.Http2ProtocolConfigurer;

import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * Allows configuring instances of {@link HttpServer}.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpServerConfigurer {

  /**
   * Sets a name to the resulting {@link HttpServer}.
   *
   * @param name the name.
   * @return this configurer
   */
  HttpServerConfigurer setName(String name);

  /**
   * Defines the host where the requests will be sent to the {@link HttpServer}. Must be provided.
   *
   * @param host where to establish the server
   * @return this configurer
   */
  HttpServerConfigurer setHost(String host);

  /**
   * Defines the port where the requests will be received by the {@link HttpServer}. Must be provided.
   *
   * @param port where to listen
   * @return this configurer
   */
  HttpServerConfigurer setPort(int port);

  /**
   * Required exclusively for HTTPS, this defines through a {@link TlsContextFactory} all the TLS related data to establish such
   * connections. Set to {@code null} by default.
   *
   * @param tlsContextFactory a {@link TlsContextFactory} with the required data.
   * @return this configurer
   */
  HttpServerConfigurer setTlsContextFactory(TlsContextFactory tlsContextFactory);

  /**
   * Defines if connections should be kept after a request is completed or closed. Default value is {@code true}.
   *
   * @param usePersistentConnections {@code boolean} specifying the decision
   * @return this configurer
   */
  HttpServerConfigurer setUsePersistentConnections(boolean usePersistentConnections);

  /**
   * Defines the number of milliseconds that a connection can remain idle before being closed. Only relevant if persistent
   * connections are used, the default value is 30 seconds.
   *
   * @param connectionIdleTimeout timeout value (in milliseconds)
   * @return this configurer
   */
  HttpServerConfigurer setConnectionIdleTimeout(int connectionIdleTimeout);

  /**
   * Defines a {@link Supplier} for a {@link Scheduler} that will be used by the {@link HttpServer} to process requests. Must be
   * specified if the server isn't associated with a flow, where a processing strategy will define the scheduling.
   *
   * @param schedulerSupplier a supplier for the scheduler.
   * @return this configurer
   */
  HttpServerConfigurer setSchedulerSupplier(Supplier<Scheduler> schedulerSupplier);

  /**
   * Defines the number of milliseconds that should be waited when reading new input.
   *
   * @param readTimeout timeout value (in milliseconds)
   * @return this configurer
   */
  HttpServerConfigurer setReadTimeout(long readTimeout);

  /**
   * Enables/disables and configures HTTP/1 specific parameters.
   *
   * @param configCallback a callback that receives the {@link Http1ProtocolConfigurer}.
   * @return this configurer
   */
  HttpServerConfigurer configHttp1(Consumer<Http1ProtocolConfigurer> configCallback);

  /**
   * Enables/disables and configures HTTP/2 specific parameters.
   *
   * @param configCallback a callback that receives the {@link Http2ProtocolConfigurer}.
   * @return this configurer
   */
  HttpServerConfigurer configHttp2(Consumer<Http2ProtocolConfigurer> configCallback);
}
