/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.queue;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.NamedObject;

import java.io.Serializable;

/**
 * Interface for mule queues used for VM.
 *
 * @since 1.0
 */
@NoImplement
public interface Queue extends NamedObject {

  /**
   * @return the number of elements in this queue.
   */
  int size();

  /**
   * Puts a new object in this queue and wait if necessary.
   */
  void put(Serializable object) throws InterruptedException;

  /**
   * Blocks and retrieves an object from this queue.
   *
   * @return an object.
   */
  Serializable take() throws InterruptedException;

  /**
   * Puts back an element in the queue, for example, after a rollback operation.
   *
   * @param item the item to untake.
   * @throws InterruptedException
   */
  void untake(Serializable item) throws InterruptedException;

  /**
   * @return the next element in the Queue.
   * @throws InterruptedException
   */
  Serializable peek() throws InterruptedException;

  /**
   * Takes an element from the queue, with a timeout on this operation.
   *
   * @param timeout (in millis) the timeout to consider for the operation.
   * @return an Object.
   * @throws InterruptedException
   */
  Serializable poll(long timeout) throws InterruptedException;

  /**
   * Puts an item to the queue, considering timeout on this operation.
   *
   * @param object  to be offered.
   * @param timeout (in millis) to consider to offer the element to the queue.
   * @return true if the element could be added to the queue.
   * @throws InterruptedException
   */
  boolean offer(Serializable object, long timeout) throws InterruptedException;

  /**
   * Discards all the elements in the queue
   *
   * @throws InterruptedException
   */
  void clear() throws InterruptedException;

  /**
   * Disposes this queue by releasing it's storage and associated memory and storage. If after disposing the queue you try go get
   * it back, you'll get a fresh new one which maintains none of the original one's data
   *
   * @throws MuleException
   * @throws InterruptedException
   */
  void dispose() throws MuleException, InterruptedException;

}
