/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.annotation.values;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import org.mule.sdk.api.annotation.DoNotEnforceMinMuleVersion;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.annotation.binding.Binding;
import org.mule.sdk.api.values.Value;
import org.mule.sdk.api.values.ValueProvider;

import java.lang.annotation.Documented;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * Marks a parameter as having a field or set of fields that have the capability of having its {@link Value values} resolved. This
 * resolution is resolved by the {@link ValueProvider} referenced in {@link FieldValues#value()}.
 *
 * @since 1.4
 */
@Target({PARAMETER, FIELD})
@Retention(RUNTIME)
@Repeatable(FieldsValues.class)
@Documented
@MinMuleVersion("4.4")
@DoNotEnforceMinMuleVersion
public @interface FieldValues {

  /**
   * Gets the target selectors of the fields of the parameter whose values are resolved by the {@link ValueProvider} referenced in
   * {@link FieldValues#value()}.
   * <p>
   * The following is an example of a possible value for the parameter so that the type of the parameter is understood:
   *
   * <pre>
   * {
   *   "routingInfo": {
   *   "channelId":"CHANNEL_ID_VALUE",
   *   "region":"SOUTH"
   *   },
   *   "message":"The message to send"
   *  }
   * </pre>
   * 
   * Using the DataWeave syntax, the target selector for the "channelId" field will be "routingInfo.channelId". In this case, the
   * format of the target selector is the name of the fields to access separated by a dot (`.`).
   * <p>
   * The following is another example using XML where the target is an attribute:
   *
   * <pre>
   * {@code
   *  <customers>
   *    <person gender="male"/>
   *      <name>John</name>
   *      <occupation>Banker</occupation>
   *    </person>
   *  </customers>
   * }
   * </pre>
   * 
   * Using the DataWeave syntax the gender attribute target selector will be "customers.person.@gender".
   *
   * <p>
   * When the given {@link ValueProvider} referenced in {@link FieldValues#value()} is a single level provider, the returned array
   * will have only one target selector. If it is multi-level, the returned array will have the selectors of the value parts in
   * order they are resolved.
   * <p>
   * The following is an example of a possible value for the parameter so that the type of the parameter is understood:
   * 
   * <pre>
   *  {
   *    "name" : "John",
   *    "occupation" : "Banker",
   *    "location": {
   *    "continent": "CONTINENT",
   *     "country": "COUNTRY",
   *     "city": "CITY"
   *     }
   *  }
   * </pre>
   * <p>
   * If a multi-level value provider is defined for the fields continent, country and city, the array returned must be
   * ["location.continent", "location.country", "location.city"]
   * <p>
   * All the values must be in the same parameter for multi-level resolution.
   *
   * @return the target selectors of the fields of the parameter
   */
  String[] targetSelectors();

  /**
   * @return the associated {@link ValueProvider} for the parameter
   */
  Class<? extends ValueProvider> value();

  /**
   * @return a boolean indicating if this values are closed or not
   */
  boolean open() default true;

  /**
   * @return bindings between parameters in the {@link OfValues#value()} to extraction expressions.
   */
  Binding[] bindings() default {};
}
