/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.connectivity;

import org.mule.sdk.api.annotation.MinMuleVersion;

/**
 * A {@link ConnectionProvider} specialization which pools connections.
 * <p>
 * This interface is declarative. The actual pooling is not implemented on the provider itself but performed by the runtime.
 * <p>
 * This interface also provides {@link #onBorrow(Object)} and {@link #onReturn(Object)} allowing to perform operations when a
 * connection is borrow and/or returned to the pool.
 *
 * @param <C> the generic type of the connections to be handled
 * @since 1.0
 */
@MinMuleVersion("4.5")
public interface PoolingConnectionProvider<C> extends ConnectionProvider<C> {

  /**
   * Executes after the {@code connection} was successfully borrowed from the pool but before it's handed over to the requester.
   * <p>
   * This method can alter the state of the {@code connection} as long as it remains on a usable state.
   *
   * @param connection the pooled connection
   */
  default void onBorrow(C connection) {}

  /**
   * Executes right before the pool accepts the {@code connection} back.
   * <p>
   * If this method throws exception, then the pool will invalidate the {@code connection}. On the other hand, a successful
   * execution of this method does not imply that the pool will successful accept the connection back. The pool is still free to
   * invalidate the connection anyway.
   * <p>
   * This method can alter the state of the {@code connection} as long as it remains on a usable state.
   *
   * @param connection the pooled connection
   */
  default void onReturn(C connection) {}

}
