/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.annotation.binding;

import org.mule.sdk.api.annotation.DoNotEnforceMinMuleVersion;
import org.mule.sdk.api.annotation.MinMuleVersion;

/**
 * Marks a binding between a parameter in a {@link org.mule.sdk.api.values.ValueProvider} or a
 * {@link org.mule.sdk.api.data.sample.SampleDataProvider} to an extraction expression. This way, the provider can take care of
 * the acquisition of the acting parameter, and it can be reused among other components which hold the acting parameter on
 * different parameters, or fields.
 */
@MinMuleVersion("4.4")
@DoNotEnforceMinMuleVersion
public @interface Binding {

  /**
   * @return The name of the field in the {@link org.mule.sdk.api.values.ValueProvider} or
   *         {@link org.mule.sdk.api.data.sample.SampleDataProvider} that is annotated with the
   *         {@link org.mule.sdk.api.annotation.param.Parameter} annotation.
   */
  String actingParameter();

  /**
   * Gets the expression used to get the acting part of the component whose value is taken for the provider acting parameter.
   *
   * This expression is a DataWeave expression that only uses selectors to retrieve the value for the acting parameter. The
   * available context for the expression has the value of all the component's parameters. The expression must start with the name
   * of a parameter of the component, then if the value comes from a part of the parameter, the expression follows with the
   * selector to the desired field. For example:
   *
   * This is an example of a possible value for a parameter named body so that the type of the parameter is understood.
   * 
   * <pre>
   *        {
   *          "routingInfo": {
   *            "channelId": "CHANNEL_ID_VALUE",
   *            "region": "SOUTH"
   *          }
   *            "message": "The message to send"
   *        }
   * </pre>
   * 
   * The extractionExpression for using the "channelId" field as acting parameter is "body.routingInfo.channelId".
   *
   * No aggregations between parameters are allowed.
   *
   * The extraction expression result will be taken as application/java. No other media type is supported for the output.
   *
   * If the value for this attribute is the name of a component parameter, it means that the value of the acting parameter, is
   * that of the whole parameter.
   * 
   * @return the expression used to extract the value of the acting parameter.
   */
  String extractionExpression();

}
