/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;

import java.io.IOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.concurrent.ExecutionException;

import io.netty.channel.ChannelHandlerContext;

public class NettyHttp1RequestReadyCallback implements HttpResponseReadyCallback {

  private final ChannelHandlerContext ctx;
  private final HttpRequest httpRequest;

  public NettyHttp1RequestReadyCallback(ChannelHandlerContext ctx, HttpRequest muleRequest) {
    this.ctx = ctx;
    this.httpRequest = muleRequest;
  }

  @Override
  public void responseReady(HttpResponse response, ResponseStatusCallback responseStatusCallback) {
    try {
      sendResponse(ctx, response, responseStatusCallback);
    } catch (IOException | ExecutionException | InterruptedException e) {
      responseStatusCallback.responseSendFailure(e);
    }
  }

  @Override
  public Writer startResponse(HttpResponse response, ResponseStatusCallback responseStatusCallback, Charset encoding) {
    return null;
  }

  private void sendResponse(ChannelHandlerContext ctx, HttpResponse response, ResponseStatusCallback callback)
      throws IOException, ExecutionException, InterruptedException {
    boolean isHeadMethod = isHeadMethod(httpRequest);
    if (isHeadMethod) {
      // Given the RFC-7231, section 4.3.2:
      // The HEAD method is identical to GET except that the server MUST NOT
      // send a message body in the response (i.e., the response terminates at
      // the end of the header section).
      // For reference: https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.2
      response = buildResponseWithEmptyBody(response);
    }
    if (response.getEntity().isStreaming()) {
      new StreamingResponseSender(httpRequest, ctx, response, callback).send();
    } else {
      new DirectResponseSender(httpRequest, ctx, response, callback).send();
    }

    if (isHeadMethod) {
      ctx.close();
    }
  }

  private static HttpResponse buildResponseWithEmptyBody(HttpResponse response) throws IOException {
    if (response.getEntity().isStreaming()) {
      response.getEntity().getContent().close();
    }
    return new HttpResponseBuilder(response).entity(new EmptyHttpEntity()).build();
  }

  private boolean isHeadMethod(HttpRequest httpRequest) {
    return httpRequest.getMethod().equals("HEAD");
  }
}
