/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.HOST;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Names.USER_AGENT;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableSet;

import static io.netty.handler.codec.http.HttpHeaderValues.CHUNKED;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_0;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpVersion;

/**
 * Utility class that converts Mule objects to the Netty equivalent.
 */
public final class MuleToNettyUtils {

  private static final Set<String> DEFAULT_EMPTY_BODY_METHODS = unmodifiableSet(new HashSet<>(asList("GET", "HEAD", "OPTIONS")));

  private MuleToNettyUtils() {
    // private constructor to avoid wrong instantiations
  }

  /**
   * Utility to transform the Mule HTTP API's {@link HttpProtocol} to the equivalent Netty's {@link HttpVersion}.
   *
   * @param protocol the Mule HTTP API value.
   * @return the Netty's value.
   */
  public static HttpVersion toNettyHttpVersion(HttpProtocol protocol) {
    switch (protocol) {
      case HTTP_0_9:
        return new HttpVersion("HTTP/0.9", false);
      case HTTP_1_0:
        return HTTP_1_0;
      case HTTP_1_1:
        return HTTP_1_1;
      default:
        return null;
    }
  }

  public static void addAllRequestHeaders(org.mule.runtime.http.api.domain.message.request.HttpRequest request,
                                          HttpHeaders currentHeaders, URI uri) {
    currentHeaders.set(HOST, buildHostHeaderValue(uri));
    currentHeaders.set(USER_AGENT, "Mule HTTP Client");

    boolean hasTransferEncoding = false;
    Collection<String> headerNames = request.getHeaderNames();
    for (String key : headerNames) {
      Collection<String> values = sanitizeValues(request.getHeaderValues(key));
      currentHeaders.set(key, values);
      if (key.equalsIgnoreCase(TRANSFER_ENCODING)) {
        hasTransferEncoding = true;
      }
    }

    if (!hasTransferEncoding && request.getProtocol().equals(HttpProtocol.HTTP_1_1)
        && !(request.getEntity() instanceof EmptyHttpEntity && DEFAULT_EMPTY_BODY_METHODS.contains(request.getMethod()))) {
      OptionalLong bytesLength = request.getEntity().getBytesLength();
      if (bytesLength.isPresent()) {
        currentHeaders.setInt(CONTENT_LENGTH, (int) bytesLength.getAsLong());
      } else {
        currentHeaders.set(TRANSFER_ENCODING, CHUNKED);
      }
    }
  }

  public static Collection<String> sanitizeValues(Collection<String> values) {
    boolean hasToSanitize = false;
    for (String value : values) {
      if (value != null && value.contains("\n")) {
        hasToSanitize = true;
        break;
      }
    }
    if (!hasToSanitize) {
      return values;
    }

    List<String> sanitized = new ArrayList<>(values.size());
    for (String value : values) {
      sanitized.add(sanitizeValue(value));
    }
    return sanitized;
  }

  public static String sanitizeValue(String value) {
    if (value != null) {
      return value.replace('\n', ' ');
    }
    return null;
  }

  public static boolean calculateShouldRemoveContentLength(org.mule.runtime.http.api.domain.message.request.HttpRequest request) {
    return DEFAULT_EMPTY_BODY_METHODS.contains(request.getMethod()) && request.getEntity() instanceof EmptyHttpEntity;
  }

  private static int getDefaultPort(String scheme) {
    if (scheme == null || scheme.equals("http")) {
      return 80;
    }

    if (scheme.equals("https")) {
      return 443;
    }

    return -1;
  }

  private static String buildHostHeaderValue(URI uri) {
    int defaultPortForScheme = getDefaultPort(uri.getScheme());

    String specifiedHost;
    if (defaultPortForScheme == uri.getPort()) {
      specifiedHost = uri.getHost();
    } else {
      specifiedHost = uri.getHost() + ":" + uri.getPort();
    }

    if (specifiedHost == null) {
      return "127.0.0.1";
    } else {
      return specifiedHost;
    }
  }
}
