/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static io.netty.buffer.ByteBufUtil.getBytes;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.service.http.netty.impl.message.BaseHttp2Request;
import org.mule.service.http.netty.impl.message.Http2FramesHandlerAdapter;
import org.mule.service.http.netty.impl.message.content.BaseHttpEntity;
import org.mule.service.http.netty.impl.server.util.DefaultServerAddress;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;

import java.io.IOException;
import java.net.InetSocketAddress;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;

public class NettyToMuleHttp2RequestHandlerAdapter extends Http2FramesHandlerAdapter {

  private final HttpListenerRegistry httpListenerRegistry;
  private final BaseHttpEntity requestContent;

  public NettyToMuleHttp2RequestHandlerAdapter(HttpListenerRegistry httpListenerRegistry) {
    this.httpListenerRegistry = httpListenerRegistry;

    // TODO: Should this implementation be somehow configurable?
    this.requestContent = new BaseHttpEntity();
  }

  @Override
  public void onHeadersFrameRead(ChannelHandlerContext ctx, Http2HeadersFrame headersFrame)
      throws IOException, InterruptedException {
    if (headersFrame.isEndStream()) {
      requestContent.close();
    }

    InetSocketAddress socketAddress = (InetSocketAddress) ctx.channel().localAddress();
    DefaultServerAddress serverAddress = new DefaultServerAddress(socketAddress.getAddress(), socketAddress.getPort());
    HttpRequest request = new BaseHttp2Request(headersFrame.headers().method().toString(),
                                               headersFrame.headers().path().toString(), requestContent);

    RequestHandler requestHandler = httpListenerRegistry.getRequestHandler(serverAddress, request);
    // TODO: Set the correct scheme
    requestHandler.handleRequest(new NettyHttpRequestContext(request, ctx, null),
                                 new NettyHttp2RequestReadyCallback(ctx, headersFrame));
  }

  @Override
  public void onDataFrameRead(ChannelHandlerContext ctx, Http2DataFrame dataFrame) throws IOException {
    ByteBuf content = dataFrame.content();
    byte[] data = getBytes(content);
    requestContent.feed(data, 0, data.length);
    if (dataFrame.isEndStream()) {
      requestContent.close();
    }
  }
}
