/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.service;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.config.MuleConfiguration;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientFactory;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.runtime.http.api.utils.RequestMatcherRegistry;
import org.mule.service.http.netty.impl.client.HttpClientConnectionManager;
import org.mule.service.http.netty.impl.server.HttpServerConnectionManager;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistryBuilder;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of {@link HttpService} that uses Netty to create {@link HttpServer}s and Reactor Netty to create
 * {@link HttpClient}s.
 */
public class NettyHttpServiceImplementation implements HttpService, Startable, Stoppable {

  static {
    System.setProperty("io.netty.handler.codec.http.defaultStrictLineParsing", "false");
  }

  private final HttpServerConnectionManager serverConnectionManager;
  private final HttpClientConnectionManager clientConnectionManager;

  private static final Logger logger = LoggerFactory.getLogger(NettyHttpServiceImplementation.class);

  public NettyHttpServiceImplementation(SchedulerService schedulerService) {
    this(schedulerService, new HttpServerConnectionManager(schedulerService), new HttpClientConnectionManager(schedulerService));
  }

  protected NettyHttpServiceImplementation(SchedulerService schedulerService,
                                           HttpServerConnectionManager serverConnectionManager,
                                           HttpClientConnectionManager clientConnectionManager) {
    this.serverConnectionManager = serverConnectionManager;
    this.clientConnectionManager = clientConnectionManager;
  }

  @Override
  public HttpServerFactory getServerFactory() {
    return new ContextHttpServerFactory(serverConnectionManager, () -> 1000L);
  }

  @Inject
  public HttpServerFactory getServerFactory(MuleConfiguration muleConfiguration) {
    return new ContextHttpServerFactory(serverConnectionManager, muleConfiguration::getShutdownTimeout);
  }

  @Override
  public HttpClientFactory getClientFactory() {
    return new ContextHttpClientFactory(clientConnectionManager);
  }

  @Override
  public RequestMatcherRegistry.RequestMatcherRegistryBuilder getRequestMatcherRegistryBuilder() {
    return new DefaultRequestMatcherRegistryBuilder();
  }

  @Override
  public String getName() {
    return "HTTP Service using Netty";
  }

  @Override
  public void start() throws MuleException {
    initialiseIfNeeded(serverConnectionManager);
    initialiseIfNeeded(clientConnectionManager);
  }

  @Override
  public void stop() throws MuleException {
    disposeIfNeeded(serverConnectionManager, logger);
    disposeIfNeeded(clientConnectionManager, logger);
  }
}
