/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.service.http.netty.impl.client.NettyHttpClient.refreshMaxClientRequestHeaders;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertTrue;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.service.http.netty.impl.message.BaseHttp2Response;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.netty.impl.util.NoOpResponseStatusCallback;
import org.mule.service.http.netty.utils.server.TestHttpServer;
import org.mule.tck.junit4.rule.DynamicPort;

import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class NettyHttpClientHeaderLimitTestCase {

  private static final int CONNECTION_IDLE_TIMEOUT = 1000;
  private static final int MAX_NUM_HEADERS_DEFAULT = 100;
  private static final String MAX_CLIENT_REQUEST_HEADERS_KEY = SYSTEM_PROPERTY_PREFIX + "http.MAX_CLIENT_REQUEST_HEADERS";

  private HttpClient client;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public TestHttpServer testServer = new TestHttpServer("localhost", serverPort.getNumber(), false);


  @Before
  public void setUp() throws Exception {
    client = NettyHttpClient.builder()
        .withConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT)
        .withUsingPersistentConnections(true)
        .build();
    client.start();

    testServer.addRequestHandler("/hello", (request, responseSender) -> {
      HttpEntity helloContent = new StringHttpEntity("Hello from server!");
      responseSender.responseReady(new BaseHttp2Response(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();
  }

  @After
  public void tearDown() {
    client.stop();
  }

  @Issue("W-15642768")
  @Test
  public void testMaxClientRequestHeaders() throws Exception {
    // maxRequestHeaders before using system property
    assertThat(NettyHttpClient.getMaxClientRequestHeaders(), equalTo(MAX_NUM_HEADERS_DEFAULT));

    System.setProperty(MAX_CLIENT_REQUEST_HEADERS_KEY, "3");
    refreshMaxClientRequestHeaders();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/test", serverPort.getNumber()))
        .method("GET")
        .addHeader("Header1", "Value1")
        .addHeader("Header2", "Value2")
        .addHeader("Header3", "Value3")
        .addHeader("Header4", "Value4")
        .build();

    try {
      client.sendAsync(httpRequest).get(); // This will throw an ExecutionException
    } catch (ExecutionException e) {
      assertTrue(e.getCause() instanceof IllegalArgumentException);
      assertThat(e.getCause().getMessage(), containsString("Exceeded max client request headers limit"));
    }
    // maxRequestHeaders after using system property
    assertThat(NettyHttpClient.getMaxClientRequestHeaders(), equalTo(3));
    System.clearProperty(MAX_CLIENT_REQUEST_HEADERS_KEY);
    refreshMaxClientRequestHeaders();
  }
}
