/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.service.http.netty.utils.TestUtils.preservingHeaderCase;
import static org.mule.service.http.netty.utils.TestUtils.sendRawRequestToServer;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.util.NoOpResponseStatusCallback;
import org.mule.service.http.netty.utils.client.TestSSLNettyClient;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class ServerHeaderCaseSensitivityTestCase extends AbstractMuleTestCase {

  private static final String A_RAW_REQUEST_WITH_HEADERS = "GET /saveRequest HTTP/1.1\n" +
      "Host: localhost\n" +
      "User-Agent: ObviouslyMe\n" +
      "HeaderWithCase: HeaderValue\n" +
      "\n";

  private HttpServer server;
  private HttpListenerRegistry httpListenerRegistry;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public TestSSLNettyClient testClient = new TestSSLNettyClient("localhost", serverPort.getNumber());

  private final BlockingQueue<HttpRequest> receivedRequests = new LinkedBlockingQueue<>();

  @Before
  public void setup()
      throws IOException, CertificateException, NoSuchAlgorithmException, CreateException, KeyManagementException {
    httpListenerRegistry = new HttpListenerRegistry();
    server = NettyHttpServer.builder()
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withShutdownTimeout(() -> 5000L)
        .withHttpListenerRegistry(httpListenerRegistry)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(httpListenerRegistry, true, 30000, null))
        .build();
    server.start();
    server.addRequestHandler("/saveRequest", (requestContext, responseCallback) -> {
      receivedRequests.add(requestContext.getRequest());
      responseCallback.responseReady(HttpResponse.builder().addHeader("HeaderWithCase", "HeaderValue").build(),
                                     new NoOpResponseStatusCallback());
    }).start();
  }

  @After
  public void tearDown() {
    server.stop().dispose();
  }

  @Test
  public void receivedRequestLosesHeaderCaseWhenNoPropertySet() throws IOException, InterruptedException {
    String rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", serverPort.getNumber());

    HttpRequest requestSeenByServer = receivedRequests.take();
    assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("host", "user-agent", "headerwithcase"));

    assertThat(rawResponse, containsString("headerwithcase"));
  }

  @Test
  public void receivedRequestPreserveHeaderCaseWhenPropertySet() {
    preservingHeaderCase(() -> {
      String rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", serverPort.getNumber());

      HttpRequest requestSeenByServer = receivedRequests.take();
      assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("Host", "User-Agent", "HeaderWithCase"));

      assertThat(rawResponse, containsString("HeaderWithCase"));
    });
  }
}
