/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.utils.client;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.getStatusByCode;

import static java.lang.Integer.parseInt;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.message.BaseHttp2Response;
import org.mule.service.http.netty.impl.message.Http2FramesHandlerAdapter;
import org.mule.service.http.netty.impl.message.content.BaseHttpEntity;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufUtil;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;

/**
 * Handles HTTP/2 stream frame responses. This is a useful approach if you specifically want to check the main HTTP/2 response
 * DATA/HEADERs, but in this example it's used purely to see whether our request (for a specific stream id) has had a final
 * response (for that same stream id).
 */
public final class TestHttp2ClientResponseHandler extends Http2FramesHandlerAdapter {

  private final CountDownLatch latch = new CountDownLatch(1);
  private final BaseHttpEntity responseContent;
  private HttpResponse response;

  public TestHttp2ClientResponseHandler() {
    responseContent = new BaseHttpEntity();
  }

  /**
   * Waits for the latch to be decremented (i.e. for an end of stream message to be received), or for the latch to expire after 5
   * seconds.
   * 
   * @return true if a successful HTTP/2 end of stream message was received.
   */
  public boolean awaitResponseSuccessfullyCompleted() {
    try {
      return latch.await(30L, TimeUnit.SECONDS);
    } catch (InterruptedException ie) {
      System.err.println("Latch exception: " + ie.getMessage());
      return false;
    }
  }

  @Override
  public void onHeadersFrameRead(ChannelHandlerContext ctx, Http2HeadersFrame headersFrame) throws IOException {
    System.out.println("Received HTTP/2 'stream' frame: " + headersFrame);
    response = new BaseHttp2Response(getStatusByCode(parseInt(headersFrame.headers().status().toString())), responseContent);
    if (headersFrame.isEndStream()) {
      responseContent.close();
      latch.countDown();
    }
  }

  @Override
  public void onDataFrameRead(ChannelHandlerContext ctx, Http2DataFrame dataFrame) throws IOException {
    System.out.println("Received HTTP/2 'stream' frame: " + dataFrame);
    byte[] frameData = ByteBufUtil.getBytes(dataFrame.content());
    responseContent.feed(frameData, 0, frameData.length);
    if (dataFrame.isEndStream()) {
      responseContent.close();
      latch.countDown();
    }
  }

  public HttpResponse getResponse() {
    return response;
  }
}
