/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.utils.client;

import static org.mule.service.http.netty.impl.util.SslContextHelper.sslContextForClient;

import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;

import java.util.concurrent.ExecutionException;

import io.netty.handler.ssl.SslContext;
import org.junit.rules.ExternalResource;

/**
 * An HTTP1 client configured with ssl that allows you to send HTTP1 Requests.
 */
public class TestSSLNettyClient extends ExternalResource {

  private static final String PASS = "mulepassword";

  private final String host;
  private final int port;
  private NettyHttpClient httpClient;

  public TestSSLNettyClient(String host, int port) {
    this.host = host;
    this.port = port;
  }

  @Override
  public void before() throws Throwable {
    initializeClient();
  }

  @Override
  public void after() {
    httpClient.stop();
  }

  public String getHost() {
    return host;
  }

  public int getPort() {
    return port;
  }

  public void reHandshake() throws Exception {
    httpClient.stop();
    initializeClient();
  }

  private void initializeClient() throws Exception {
    TlsContextFactory tlsContextFactory =
        TlsContextFactory.builder().trustStorePath("trustStore").trustStorePassword(PASS).build();

    SslContext sslContext = sslContextForClient(tlsContextFactory);
    httpClient = NettyHttpClient.builder().withSslContext(sslContext).build();
    httpClient.start();
  }

  public HttpResponse sendGet(String path) throws ExecutionException, InterruptedException {
    return this.sendGet(path, MultiMap.emptyMultiMap());
  }

  public HttpResponse sendGet(String path, MultiMap<String, String> headers) throws ExecutionException, InterruptedException {
    HttpRequest request = HttpRequest.builder()
        .protocol(HttpProtocol.HTTP_1_1)
        .method("GET")
        .uri(buildUriString(path))
        .headers(headers)
        .build();
    return httpClient.sendAsync(request, HttpRequestOptions.builder().build()).get();
  }

  public HttpResponse sendPost(String path, String content) throws ExecutionException, InterruptedException {
    HttpRequest request = HttpRequest.builder()
        .protocol(HttpProtocol.HTTP_1_1)
        .method("POST")
        .uri(buildUriString(path))
        .entity(new StringHttpEntity(content))
        .build();
    return httpClient.sendAsync(request, HttpRequestOptions.builder().build()).get();
  }

  private String buildUriString(String path) {
    return String.format("https://%s:%d%s%s", host, port, path.startsWith("/") ? "" : "/", path);
  }

  public NettyHttpClient getNettyClient() {
    return this.httpClient;
  }
}
