/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.utils.server;

import static org.mule.runtime.http.api.server.MethodRequestMatcher.acceptAll;
import static org.mule.service.http.netty.impl.util.SslContextHelper.sslContextForServer;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.PathAndMethodRequestMatcher;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;

import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import io.netty.handler.ssl.SslContext;
import org.junit.rules.ExternalResource;

public class TestHttpServer extends ExternalResource {

  private static final String PASS = "mulepassword";
  private final String host;
  private final int port;
  private final boolean useSsl;
  private final HttpListenerRegistry httpListenerRegistry;
  private HttpServer httpServer;

  public TestHttpServer(String host, int port, boolean useSsl) {
    this.host = host;
    this.port = port;
    this.useSsl = useSsl;
    this.httpListenerRegistry = new HttpListenerRegistry();
  }

  @Override
  protected void before() throws Throwable {
    SslContext sslContext = createSslContext();
    httpServer = NettyHttpServer.builder()
        .withSslContext(sslContext)
        .withShutdownTimeout(() -> 5000L)
        .withServerAddress(new InetSocketAddress(host, port))
        .withHttpListenerRegistry(httpListenerRegistry)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(httpListenerRegistry, true, 30000,
                                                                           sslContext))
        .build();
    httpServer.start();
  }

  @Override
  protected void after() {
    httpServer.stop().dispose();
  }

  public SslContext createSslContext() throws NoSuchAlgorithmException, KeyManagementException, CreateException {
    if (!useSsl) {
      return null;
    }

    TlsContextFactory tlsContextFactory = TlsContextFactory.builder()
        .keyStorePath("serverKeystore")
        .keyStorePassword(PASS)
        .keyPassword(PASS)
        .build();
    return sslContextForServer(tlsContextFactory);
  }

  public RequestHandlerManager addRequestHandler(String path, RequestHandler requestHandler) {
    return httpListenerRegistry.addRequestHandler(httpServer, requestHandler, PathAndMethodRequestMatcher.builder()
        .methodRequestMatcher(acceptAll())
        .path(path)
        .build());
  }

  public void removeAllHandlers() {
    httpListenerRegistry.removeHandlersFor(httpServer);
  }
}
