/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NO_CONTENT;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.emptyString;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

@Issue("W-15631414")
public class NoContentTestCase extends AbstractMuleTestCase {

  private HttpClient client;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public HardcodedResponseTcpServer server = new HardcodedResponseTcpServer(serverPort.getNumber());

  @Before
  public void setUp() {
    client = NettyHttpClient.builder().build();
    client.start();

    server.setResponse("HTTP/1.1 204 No Content\r\n\r\nGarbage");
  }

  @Test
  public void responseWithEmptyContentFor204StatusCode() throws IOException, ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder(PRESERVE_HEADER_CASE)
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(NO_CONTENT.getStatusCode()));
    assertThat(response.getReasonPhrase(), is(NO_CONTENT.getReasonPhrase()));

    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is(emptyString()));
  }

}
