/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.utils.client;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.util.concurrent.TimeUnit;

import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.http2.DefaultHttp2DataFrame;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.DefaultHttp2HeadersFrame;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;
import io.netty.handler.ssl.ApplicationProtocolConfig;
import io.netty.handler.ssl.ApplicationProtocolConfig.Protocol;
import io.netty.handler.ssl.ApplicationProtocolConfig.SelectedListenerFailureBehavior;
import io.netty.handler.ssl.ApplicationProtocolConfig.SelectorFailureBehavior;
import io.netty.handler.ssl.ApplicationProtocolNames;
import io.netty.handler.ssl.IdentityCipherSuiteFilter;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.SslProvider;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import org.junit.rules.ExternalResource;

/**
 * An HTTP2 client that allows you to send HTTP2 frames to a server using the newer HTTP2 approach (via
 * {@link io.netty.handler.codec.http2.Http2FrameCodec}).
 */
// TODO: Should include {@link io.netty.handler.codec.http2.Http2ClientUpgradeCodec} if the HTTP/2 server you are
// hitting doesn't support h2c/prior knowledge.
public class TestHttp2Client extends ExternalResource {

  private final String host;
  private final int port;
  private boolean connected;

  private Bootstrap bootstrap;

  private Channel connectionChannel;
  private EventLoopGroup clientWorkerGroup;
  private SslContext sslContext;

  public TestHttp2Client(String host, int port) {
    this.host = host;
    this.port = port;
    this.connected = false;
  }

  @Override
  protected void before() throws Throwable {
    clientWorkerGroup = new NioEventLoopGroup();
    final SslProvider provider =
        SslProvider.isAlpnSupported(SslProvider.OPENSSL) ? SslProvider.OPENSSL : SslProvider.JDK;
    sslContext = SslContextBuilder.forClient()
        .sslProvider(provider)
        .ciphers(null, IdentityCipherSuiteFilter.INSTANCE)
        // you probably won't want to use this in production, but it is fine for this example:
        .trustManager(InsecureTrustManagerFactory.INSTANCE)
        .applicationProtocolConfig(new ApplicationProtocolConfig(
                                                                 Protocol.ALPN,
                                                                 SelectorFailureBehavior.NO_ADVERTISE,
                                                                 SelectedListenerFailureBehavior.ACCEPT,
                                                                 ApplicationProtocolNames.HTTP_2,
                                                                 ApplicationProtocolNames.HTTP_1_1))
        .build();

    bootstrap = new Bootstrap();
    bootstrap.group(clientWorkerGroup)
        .channel(NioSocketChannel.class)
        .option(ChannelOption.SO_KEEPALIVE, true)
        .remoteAddress(host, port);
    // .handler(new Http2ClientInitializer(sslContext));
  }

  @Override
  protected void after() {
    try {
      if (connected) {
        // Wait until the connection is closed.
        connectionChannel.close().syncUninterruptibly();
      }
    } finally {
      clientWorkerGroup.shutdownGracefully();
    }
  }

  private void connectIfNeeded() {
    if (!connected) {
      connectionChannel = bootstrap.connect().syncUninterruptibly().channel();
      System.out.println("Connected to [" + host + ':' + port + ']');
      connected = true;
    }
  }

  public String getHost() {
    return host;
  }

  public int getPort() {
    return port;
  }

  public HttpResponse sendGet(String path) {
    connectIfNeeded();

    // This is the way to "connect" a stream: one stream <--> one request.
    TestHttp2StreamHandler streamHandler = new TestHttp2StreamHandler(connectionChannel);

    // Send request (a HTTP/2 HEADERS frame - with ':method = GET' in this case)
    final DefaultHttp2Headers headers = new DefaultHttp2Headers();
    headers.method("GET");
    headers.path(path);
    headers.scheme("https");
    final Http2HeadersFrame headersFrame = new DefaultHttp2HeadersFrame(headers, true);
    ChannelFuture writeFuture = streamHandler.writeAndFlush(headersFrame);
    System.out.println("Sent HTTP/2 GET request to '" + path + "'");

    return tryGetResponse(streamHandler, writeFuture);
  }

  public HttpResponse sendPost(String path, String content) {
    connectIfNeeded();
    TestHttp2StreamHandler streamHandler = new TestHttp2StreamHandler(connectionChannel);

    final DefaultHttp2Headers headers = new DefaultHttp2Headers();
    headers.method("POST");
    headers.path(path);
    headers.scheme("https");
    final Http2HeadersFrame headersFrame = new DefaultHttp2HeadersFrame(headers, false);
    streamHandler.write(headersFrame);

    ByteBuf requestContent = streamHandler.allocateByteBuf();
    ByteBufUtil.writeUtf8(requestContent, content);
    final Http2DataFrame dataFrame = new DefaultHttp2DataFrame(requestContent, true);
    ChannelFuture writeFuture = streamHandler.writeAndFlush(dataFrame);

    return tryGetResponse(streamHandler, writeFuture);
  }

  private HttpResponse tryGetResponse(TestHttp2StreamHandler streamHandler, ChannelFuture writeFuture) {
    // Wait for the responses (or for the latch to expire), then clean up the connections
    if (!writeFuture.awaitUninterruptibly(10, TimeUnit.SECONDS)) {
      throw new IllegalStateException("Timed out waiting to write");
    }
    if (!writeFuture.isSuccess()) {
      throw new RuntimeException(writeFuture.cause());
    }
    if (!streamHandler.awaitResponseSuccessfullyCompleted()) {
      System.err.println("Did not get HTTP/2 response in expected time.");
      return null;
    }

    HttpResponse response = streamHandler.getResponse();
    System.out.println("Finished HTTP/2 request. Got response: " + response);
    return response;
  }
}
