/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.scheduler.SchedulerConfig.config;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.service.http.netty.impl.util.SslContextHelper.sslContextForServer;

import static java.lang.Integer.getInteger;
import static java.lang.Integer.max;
import static java.lang.Runtime.getRuntime;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerAlreadyExistsException;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.server.util.HttpServerAdapter;

import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import io.netty.handler.ssl.SslContext;

public class HttpServerConnectionManager implements ContextHttpServerConnectionFactory {

  private static final int SERVER_SELECTOR_THREAD_COUNT =
      getInteger("mule.http.server.selectors.count", max(getRuntime().availableProcessors(), 2));
  private final Map<String, HttpServer> servers = new ConcurrentHashMap<>();
  private final SchedulerService schedulerService;

  public HttpServerConnectionManager(SchedulerService schedulerService) {
    this.schedulerService = schedulerService;
  }

  protected HttpServerConnectionManager() {
    this(null);
  }

  @Override
  public HttpServer create(HttpServerConfiguration configuration, Supplier<Long> shutdownTimeout) throws ServerCreationException {
    String name = configuration.getName();
    HttpServer alreadyPresent = servers.get(name);
    if (alreadyPresent != null) {
      throw new ServerAlreadyExistsException(alreadyPresent.getServerAddress());
    }

    HttpServer created = createServer(name, configuration, shutdownTimeout);
    servers.putIfAbsent(name, created);
    return created;
  }

  @Override
  public HttpServer lookup(String name) throws ServerNotFoundException {
    HttpServer server = servers.get(name);
    if (server == null) {
      throw new ServerNotFoundException(name);
    } else {
      return new NoLifecycleHttpServer(server);
    }
  }

  @Override
  public HttpServer getOrCreateServer(String name, Supplier<? extends HttpServerConfiguration> configuration,
                                      Supplier<Long> shutdownTimeout)
      throws ServerCreationException {
    checkArgument(name != null, "Server name can't be null");
    try {
      return servers.computeIfAbsent(name, serverName -> createServer(name, configuration.get(), shutdownTimeout));
    } catch (MuleRuntimeException e) {
      throw new ServerCreationException(e.getMessage(), e);
    }
  }

  private HttpServer createServer(String name, HttpServerConfiguration configuration, Supplier<Long> shutdownTimeout)
      throws MuleRuntimeException {
    try {
      checkArgument(name != null, "Server name can't be null");
      checkArgument(configuration != null, "Server configuration can't be null");
      HttpListenerRegistry httpListenerRegistry = new HttpListenerRegistry();
      SslContext sslContext = sslContextForServer(configuration.getTlsContextFactory());
      Scheduler selectorsScheduler = createSelectorsScheduler();
      NettyHttpServer.Builder builder = NettyHttpServer.builder()
          // TODO: Add the other parameters
          .withServerAddress(new InetSocketAddress(configuration.getHost(), configuration.getPort()))
          .withHttpListenerRegistry(httpListenerRegistry)
          .withSslContext(sslContext)
          .withSelectorsScheduler(selectorsScheduler)
          .withSelectorsCount(SERVER_SELECTOR_THREAD_COUNT)
          .withShutdownTimeout(shutdownTimeout)
          .doOnDispose(() -> servers.remove(name));
      return enrichServerBuilder(builder, httpListenerRegistry, configuration, sslContext).build();
    } catch (IllegalArgumentException | NoSuchAlgorithmException | KeyManagementException e) {
      throw new MuleRuntimeException(e);
    }
  }

  private Scheduler createSelectorsScheduler() {
    if (schedulerService == null) {
      return null;
    }
    return schedulerService
        .customScheduler(config().withMaxConcurrentTasks(SERVER_SELECTOR_THREAD_COUNT).withName("http.listener"), 0);
  }

  protected NettyHttpServer.Builder enrichServerBuilder(NettyHttpServer.Builder builder,
                                                        HttpListenerRegistry listenerRegistry,
                                                        HttpServerConfiguration configuration,
                                                        SslContext sslContext) {
    return builder
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, configuration, sslContext));
  }

  /**
   * Wrapper that avoids all lifecycle operations, so that the inner server owns that exclusively.
   */
  private static class NoLifecycleHttpServer extends HttpServerAdapter {

    public NoLifecycleHttpServer(HttpServer delegate) {
      super(delegate);
    }

    @Override
    public HttpServer start() {
      return this;
    }

    @Override
    public HttpServer stop() {
      return this;
    }

    @Override
    public void dispose() {
      // Do nothing
    }
  }
}
