/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.proxy;

import static reactor.netty.NettyPipeline.HttpCodec;

import org.mule.runtime.http.api.client.proxy.ProxyConfig;

import java.net.InetSocketAddress;
import java.net.SocketAddress;

import io.netty.channel.ChannelPipeline;

/**
 * Configures a {@link ChannelPipeline} with the corresponding (if any) to the given {@link ProxyConfig}. The handler could be:
 * <il>
 * <li>none, if the config is null or the host where the client is connecting is in the list of non-proxy-hosts.</li>
 * <li>a {@link BlindTunnelingProxyClientHandler}, if the host is not a "non-proxy-host" and the user passes
 * useTunneling=true.</li>
 * <li>a {@link MessageForwardingProxyClientHandler}, if the host is not a "non-proxy-host" and the user passes
 * useTunneling=false.</li> </il>
 */
public class ProxyPipelineConfigurer {

  private final ProxyConfig proxyConfig;

  public ProxyPipelineConfigurer(ProxyConfig proxyConfig) {
    this.proxyConfig = proxyConfig;
  }

  public void configurePipeline(ChannelPipeline pipeline, SocketAddress remoteAddress, boolean useTunneling) {
    if (proxyConfig == null) {
      // No proxy.
      return;
    }

    if (isNonProxyHost(proxyConfig.getNonProxyHosts(), remoteAddress)) {
      // Proxy configured, but not for this particular host.
      return;
    }

    if (useTunneling) {
      // Establish a blind forwarding proxy tunnel (with CONNECT method).
      // See RFC https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.6
      pipeline.addFirst("TunnelProxyHandler", new BlindTunnelingProxyClientHandler(proxyConfig));
      return;
    }

    // Otherwise, we use a message-forwarding HTTP proxy agent
    // See RFC https://datatracker.ietf.org/doc/html/rfc7230#page-10
    pipeline.addAfter(HttpCodec, "ProxyAuthHandler", new MessageForwardingProxyClientHandler(proxyConfig));
  }

  private boolean isNonProxyHost(String nonProxyHosts, SocketAddress socketAddress) {
    if (nonProxyHosts == null || nonProxyHosts.isEmpty()) {
      return false;
    }

    if (!(socketAddress instanceof InetSocketAddress)) {
      // this should never happen, but if we don't know that it's a non-proxy-host, then we return false.
      return false;
    }

    final String hostString = ((InetSocketAddress) socketAddress).getHostString();
    for (String nonProxyHost : nonProxyHosts.split(",")) {
      if (nonProxyHost.trim().equals(hostString)) {
        return true;
      }
    }

    return false;
  }
}
