/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message.content;


import static org.mule.runtime.api.util.IOUtils.toByteArray;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.util.Collection;
import java.util.Optional;
import java.util.OptionalLong;

public class BaseHttpEntity implements Feedable, HttpEntity {

  private static final int ENTITY_PIPE_SIZE = 102400;

  private final PipedOutputStream sinkForReceivedData;
  private final PipedInputStream asInputStream;

  private boolean isClosed;
  private long partialLength;

  public BaseHttpEntity() {
    sinkForReceivedData = new PipedOutputStream();
    try {
      asInputStream = new PipedInputStream(sinkForReceivedData, ENTITY_PIPE_SIZE);
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
    partialLength = 0;
    isClosed = false;
  }

  @Override
  public void feed(byte[] data, int offset, int size) throws IOException {
    sinkForReceivedData.write(data, offset, size);
    partialLength += size;
  }

  @Override
  public void close() throws IOException {
    sinkForReceivedData.close();
    isClosed = true;
  }

  @Override
  public boolean isClosed() {
    return isClosed;
  }

  // @Override
  public boolean isEmpty() {
    return isClosed && partialLength == 0;
  }

  @Override
  public boolean isStreaming() {
    return false;
  }

  @Override
  public boolean isComposed() {
    return false;
  }

  @Override
  public InputStream getContent() {
    return asInputStream;
  }

  @Override
  public byte[] getBytes() throws IOException {
    return toByteArray(this.asInputStream);
  }

  @Override
  public Collection<HttpPart> getParts() throws IOException {
    return null;
  }

  @Override
  public Optional<Long> getLength() {
    if (isClosed) {
      return Optional.of(partialLength);
    } else {
      return Optional.empty();
    }
  }

  @Override
  public OptionalLong getBytesLength() {
    if (isClosed) {
      return OptionalLong.of(partialLength);
    } else {
      return OptionalLong.empty();
    }
  }
}
