/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.util.List;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class ClientBodySemanticsTestCase extends AbstractMuleTestCase {

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public HardcodedResponseTcpServer server = new HardcodedResponseTcpServer(serverPort.getNumber());
  private HttpClient client;

  @Before
  public void setUp() {
    client = NettyHttpClient.builder().build();
    client.start();

    server.setResponse("HTTP/1.1 200 OK\n" + "Content-Length: 0\n" + "\n");
  }

  @After
  public void tearDown() {
    if (client != null) {
      client.stop();
    }
  }

  @Test
  @Issue("W-15631517")
  public void noContentLengthHeaderForGetWithoutBodySemantics() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("GET")
        .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is not present in the request
    assertThat(rawRequest, not(containsString("content-length: 0")));
  }

  @Test
  @Issue("W-15631517")
  public void noContentLengthHeaderForDeleteWithoutBodySemantics() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("DELETE")
            .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is not present in the request
    assertThat(rawRequest, (containsString("content-length: 0")));
  }

  @Test
  @Issue("W-15631517")
  public void contentLengthHeaderForPostWithEmptyBody() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/hello", serverPort.getNumber())).method("POST")
            .entity(new EmptyHttpEntity()).build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "content-length: 0" is present in the request for POST
    assertThat(rawRequest, containsString("content-length: 0"));
  }

  @Test
  @Issue("W-15631517")
  public void contentLengthHeaderForPutWithEmptyBody() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("PUT")
        .entity(new EmptyHttpEntity()) // Sending empty body
        .build();

    HttpResponse ignored = client.sendAsync(httpRequest).get();
    List<String> rawRequests = server.getReceivedRawRequests();
    assertThat(rawRequests, hasSize(1));
    String rawRequest = rawRequests.get(0);
    // Ensure "Content-Length: 0" is present in the request for PUT, case-insensitively
    assertThat(rawRequest.toLowerCase(), containsString("content-length: 0"));
  }
}
