/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.functional.junit4.matchers.ThrowableCauseMatcher.hasCause;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerAlreadyExistsException;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.util.function.Supplier;

import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class HttpServerConnectionManagerTestCase extends AbstractMuleTestCase {

  public static final String TEST_HOST = "localhost";

  @Rule
  public DynamicPort dynamicPort = new DynamicPort("port");

  private HttpServerConnectionManager httpServerConnectionManager;
  private HttpServerConfiguration mockConfiguration;
  private Supplier<HttpServerConfiguration> mockConfigurationSupplier;
  private final Supplier<Long> shutdownTimeout = () -> 5000L;

  @Rule
  public ExpectedException expected = none();

  @Before
  public void setup() throws IOException, CreateException {
    httpServerConnectionManager = new HttpServerConnectionManager();

    TlsContextFactory tlsContextFactory = TlsContextFactory.builder().buildDefault();

    mockConfigurationSupplier = spy(mock(Supplier.class));
    mockConfiguration = spy(mock(HttpServerConfiguration.class));
    when(mockConfiguration.getName()).thenReturn("mockServerName");
    when(mockConfiguration.getHost()).thenReturn(TEST_HOST);
    when(mockConfiguration.getPort()).thenReturn(dynamicPort.getNumber());
    when(mockConfiguration.getTlsContextFactory()).thenReturn(tlsContextFactory);
    when(mockConfigurationSupplier.get()).thenReturn(mockConfiguration);
  }

  @Test
  public void getOrCreateServerReturnsSameInstanceIfCalledTwiceWithSameName() throws ServerCreationException {
    HttpServer firstCallServer =
        httpServerConnectionManager.getOrCreateServer("ServerName", mockConfigurationSupplier, shutdownTimeout);
    HttpServer secondCallServer =
        httpServerConnectionManager.getOrCreateServer("ServerName", mockConfigurationSupplier, shutdownTimeout);

    assertThat(secondCallServer, is(sameInstance(firstCallServer)));
  }

  @Test
  public void getOrCreateServerCallsConfigurationSupplierOnlyOnceIfCalledTwiceWithSameName() throws ServerCreationException {
    httpServerConnectionManager.getOrCreateServer("ServerName", mockConfigurationSupplier, shutdownTimeout);
    httpServerConnectionManager.getOrCreateServer("ServerName", mockConfigurationSupplier, shutdownTimeout);

    verify(mockConfigurationSupplier, times(1)).get();
  }

  @Test
  public void getOrCreateServerReturnsDifferentInstancesIfCalledTwiceWithDifferentNames() throws ServerCreationException {
    HttpServer firstCallServer =
        httpServerConnectionManager.getOrCreateServer("SomeServerName", mockConfigurationSupplier, shutdownTimeout);
    HttpServer secondCallServer =
        httpServerConnectionManager.getOrCreateServer("OtherServerName", mockConfigurationSupplier, shutdownTimeout);

    assertThat(secondCallServer, is(not(sameInstance(firstCallServer))));
  }

  @Test
  public void getOrCreateServerCallsConfigurationSupplierTwiceIfCalledTwiceWithDifferentNames()
      throws ServerCreationException {
    httpServerConnectionManager.getOrCreateServer("SomeServerName", mockConfigurationSupplier, shutdownTimeout);
    httpServerConnectionManager.getOrCreateServer("OtherServerName", mockConfigurationSupplier, shutdownTimeout);

    verify(mockConfigurationSupplier, times(2)).get();
  }

  @Test
  public void serverNameCantBeNull() throws ServerCreationException {
    expected.expect(IllegalArgumentException.class);
    expected.expectMessage("Server name can't be null");
    httpServerConnectionManager.getOrCreateServer(null, mockConfigurationSupplier, shutdownTimeout);
  }

  @Test
  public void suppliedConfigurationCantBeNull() throws ServerCreationException {
    when(mockConfigurationSupplier.get()).thenReturn(null);

    expected.expect(ServerCreationException.class);
    expected.expectCause(hasCause(instanceOf(IllegalArgumentException.class)));
    expected.expectMessage("Could not create server: java.lang.IllegalArgumentException: Server configuration can't be null");
    httpServerConnectionManager.getOrCreateServer("SomeName", mockConfigurationSupplier, shutdownTimeout);
  }

  @Test
  @Issue("W-15784799")
  public void ifAServerWasCreatedThenItCanBeLookedUp() throws ServerCreationException, ServerNotFoundException {
    httpServerConnectionManager.create(mockConfiguration, shutdownTimeout);
    HttpServer lookedUp = httpServerConnectionManager.lookup("mockServerName");
    assertThat(lookedUp, is(notNullValue()));
  }

  @Test
  @Issue("W-15784799")
  public void isAServerIsAlreadyCreatedThenCreatingItAgainFails() throws ServerCreationException {
    httpServerConnectionManager.create(mockConfiguration, shutdownTimeout);

    String expectedErrorMessage =
        format("Could not create server: A server in port(%s) already exists for host(127.0.0.1) or one overlapping it (0.0.0.0).",
               dynamicPort.getNumber());

    expected.expect(ServerAlreadyExistsException.class);
    expected.expectMessage(expectedErrorMessage);
    httpServerConnectionManager.create(mockConfiguration, shutdownTimeout);
  }

  @Test
  @Issue("W-15795503")
  public void lifecycleCantBeAppliedIfTheServerIsLookedUp() throws ServerCreationException, ServerNotFoundException, IOException {
    HttpServer created = httpServerConnectionManager.create(mockConfiguration, shutdownTimeout).start();
    HttpServer lookedUp = httpServerConnectionManager.lookup("mockServerName");

    lookedUp.stop();
    assertThat(created.isStopped(), is(false));
    assertThat(lookedUp.isStopped(), is(false));

    created.stop();
    assertThat(created.isStopped(), is(true));
    assertThat(lookedUp.isStopped(), is(true));
  }
}
