/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static io.netty.handler.codec.http.HttpHeaderValues.CLOSE;
import static io.netty.handler.codec.http.HttpHeaderValues.KEEP_ALIVE;

import io.netty.channel.ChannelDuplexHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelOutboundHandlerAdapter;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.util.AttributeKey;

/**
 * This handler removes the content length if an attribute is set in the context of the request. We have to do this to preserve
 * backwards compatibility because there are some cases that we don't expect the content length to be set (when the method does
 * not require it, we are not forcing it with a sendBodyMode ALWAYS property and the body is empty) but Reactor-netty library
 * always sets the content length because the RFC does not mandate it to be sent (it has a SHOULD NOT)
 */
public class RemoveContentLengthHandler extends ChannelOutboundHandlerAdapter {

  @Override
  public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) {
    boolean removeContentLength =
        (Boolean) ctx.attr(AttributeKey.valueOf("removeContentLength")).get();
    if (removeContentLength && msg instanceof io.netty.handler.codec.http.HttpRequest) {
      ((io.netty.handler.codec.http.HttpRequest) msg).headers()
          .remove(HttpHeaderNames.CONTENT_LENGTH);
    }
    ctx.write(msg, promise);
  }
}
