/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.message;

/**
 * Abstract class representing an NTLM message.
 * <p>
 * This class provides constants and methods for handling NTLM messages, including reading and writing various data types and
 * managing flags.
 * </p>
 * <p>
 * This implementation is based on the jcifs library, available at <a href="https://github.com/kevinboone/jcifs">jcifs GitHub
 * Repository</a>.
 * </p>
 */
public abstract class NtlmMessage {

  protected static final byte[] NTLMSSP_SIGNATURE = new byte[] {
      (byte) 'N', (byte) 'T', (byte) 'L', (byte) 'M', (byte) 'S', (byte) 'S', (byte) 'P', (byte) 0
  };

  protected static final int NTLMSSP_TYPE1 = 0x1;
  protected static final int NTLMSSP_TYPE2 = 0x2;
  protected static final int NTLMSSP_TYPE3 = 0x3;

  private static final String OEM_ENCODING = "Cp850";

  protected static final String UNI_ENCODING = "UTF-16LE";

  protected static final byte[] NTLMSSP_VERSION = new byte[] {
      6, 1, 0, 0, 0, 0, 0, 15
  };

  private int flags;

  public NtlmMessage() {}

  public NtlmMessage(int flags) {
    this.flags = flags;
  }

  static void writeULong(byte[] dest, int offset, int ulong) {
    dest[offset] = (byte) (ulong & 0xff);
    dest[offset + 1] = (byte) (ulong >> 8 & 0xff);
    dest[offset + 2] = (byte) (ulong >> 16 & 0xff);
    dest[offset + 3] = (byte) (ulong >> 24 & 0xff);
  }

  static int writeSecurityBuffer(byte[] dest, int offset, byte[] src) {
    int length = (src != null) ? src.length : 0;
    if (length == 0) {
      return offset + 4;
    }
    writeUShort(dest, offset, length);
    writeUShort(dest, offset + 2, length);
    return offset + 4;
  }

  static void writeUShort(byte[] dest, int offset, int ushort) {
    dest[offset] = (byte) (ushort & 0xff);
    dest[offset + 1] = (byte) (ushort >> 8 & 0xff);
  }

  static int writeSecurityBufferContent(byte[] dest, int pos, int off, byte[] src) {
    writeULong(dest, off, pos);
    if (src != null && src.length > 0) {
      System.arraycopy(src, 0, dest, pos, src.length);
      return src.length;
    }
    return 0;
  }

  static int readULong(byte[] src, int index) {
    return (src[index] & 0xff) | ((src[index + 1] & 0xff) << 8) | ((src[index + 2] & 0xff) << 16)
        | ((src[index + 3] & 0xff) << 24);
  }

  static byte[] readSecurityBuffer(byte[] src, int index) {
    int length = readUShort(src, index);
    int offset = readULong(src, index + 4);
    byte[] buffer = new byte[length];
    System.arraycopy(src, offset, buffer, 0, length);
    return buffer;
  }

  static int readUShort(byte[] src, int index) {
    return (src[index] & 0xff) | ((src[index + 1] & 0xff) << 8);
  }

  static String getOEMEncoding() {
    return OEM_ENCODING;
  }

  public boolean getFlag(int flag) {
    return (getFlags() & flag) != 0;
  }

  public int getFlags() {
    return flags;
  }

  public void setFlag(int flag, boolean value) {
    setFlags(value ? (getFlags() | flag) : (getFlags() & (0xffffffff ^ flag)));
  }

  public void setFlags(int flags) {
    this.flags = flags;
  }

}
