/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.proxy;

import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.service.http.netty.impl.client.auth.AuthHeaderFactory;
import org.mule.service.http.netty.impl.client.auth.BasicAuthHeaderFactory;
import org.mule.service.http.netty.impl.client.auth.ntlm.NtlmMessageFactory;

/**
 * Utility wrapper of the {@link AuthHeaderFactory} corresponding to a given {@link ProxyConfig}. It supports basic and NTLM
 * authentication schemes, and can also be used as a "null object" (it would be an already finished authenticator that returns
 * "none" as auth scheme).
 */
public class ProxyAuthenticator {

  private static final String AUTH_BASIC = "basic";
  private static final String AUTH_NONE = "none";
  private static final String AUTH_NTLM = "ntlm";

  private final String authScheme;
  private final AuthHeaderFactory authHeaderFactory;

  public ProxyAuthenticator(ProxyConfig proxyConfig) {
    if (proxyConfig.getUsername() == null) {
      this.authScheme = AUTH_NONE;
      this.authHeaderFactory = null;
      return;
    }

    if (proxyConfig instanceof ProxyConfig.NtlmProxyConfig) {
      ProxyConfig.NtlmProxyConfig ntlmProxyConfig = (ProxyConfig.NtlmProxyConfig) proxyConfig;
      this.authScheme = AUTH_NTLM;
      this.authHeaderFactory = new NtlmMessageFactory(ntlmProxyConfig.getNtlmDomain(), null, ntlmProxyConfig.getUsername(),
                                                      ntlmProxyConfig.getPassword());
      return;
    }

    this.authScheme = AUTH_BASIC;
    this.authHeaderFactory = new BasicAuthHeaderFactory(true, proxyConfig.getUsername(), proxyConfig.getPassword());
  }

  /**
   * @return the authentication scheme. It can be {@code "basic"}, {@code "ntlm"}, or {@code "none"}.
   */
  public String getAuthScheme() {
    return authScheme;
  }

  /**
   * For the special case of a "none" authenticator, this method always returns {@code true}. For the other schemes, see
   * {@link AuthHeaderFactory#hasFinished()}.
   * 
   * @return {@code true} if the authentication has finished.
   * @see AuthHeaderFactory#hasFinished().
   */
  public boolean hasFinished() {
    if (authHeaderFactory == null) {
      return true;
    }
    return authHeaderFactory.hasFinished();
  }

  /**
   * For the special case of a "none" authenticator, this method always returns {@code null}. For the other schemes, see
   * {@link AuthHeaderFactory#getNextHeader(String)}}.
   * 
   * @param receivedAuthenticateHeader the "Proxy-Authenticate" header sent by the server.
   * @return the "Proxy-Authorization" header that should be sent by the client.
   * @throws Exception if an error occurs while calculating the header value.
   */
  public String getNextHeader(String receivedAuthenticateHeader) throws Exception {
    if (authHeaderFactory == null) {
      return null;
    }
    return authHeaderFactory.getNextHeader(receivedAuthenticateHeader);
  }
}
