/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;

import static io.netty.handler.codec.DecoderResult.failure;
import static io.netty.handler.codec.http.HttpMethod.POST;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static io.netty.handler.codec.http.LastHttpContent.EMPTY_LAST_CONTENT;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.ServerAddress;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;

import java.net.InetSocketAddress;

import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.DecoderException;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.ssl.SslHandler;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

public class ForwardingToListenerHandlerTestCase {

  private ForwardingToListenerHandler handler;

  @Rule
  public MockitoRule rule = MockitoJUnit.rule();

  @Mock
  private HttpListenerRegistry httpListenerRegistry;

  @Mock
  private RequestHandler requestHandler;

  @Mock
  private RequestHandler errorRequestHandler;

  @Mock
  private SslHandler sslHandler;

  @Mock
  private ChannelHandlerContext ctx;

  @Mock
  private Channel channel;

  @Before
  public void setUp() {
    when(ctx.channel()).thenReturn(channel);

    when(channel.localAddress()).thenReturn(new InetSocketAddress("localhost", 54321));

    when(httpListenerRegistry.getRequestHandler(any(ServerAddress.class), any(HttpRequest.class))).thenReturn(requestHandler);
    when(httpListenerRegistry.getErrorHandler(any(Throwable.class))).thenReturn(errorRequestHandler);

    handler = new ForwardingToListenerHandler(httpListenerRegistry, sslHandler);
  }

  @Test
  public void whenExceptionCaughtThenContextIsClosed() throws Exception {
    handler.exceptionCaught(ctx, new Throwable("Expected"));
    verify(ctx).close();
  }

  @Test
  public void handleRequestInMultipleObjects() throws Exception {
    DefaultHttpRequest request = new DefaultHttpRequest(HTTP_1_1, POST, "https://www.mulesoft.com");
    request.headers().addInt(CONTENT_LENGTH, 12);

    handler.channelRead0(ctx, request);

    ArgumentCaptor<HttpRequestContext> reqCtxCaptor = forClass(HttpRequestContext.class);
    verify(requestHandler).handleRequest(reqCtxCaptor.capture(), any(HttpResponseReadyCallback.class));

    HttpRequest muleRequest = reqCtxCaptor.getValue().getRequest();
    HttpEntity entity = muleRequest.getEntity();

    byte[] bytes = new byte[4];

    handler.channelRead0(ctx, new DefaultHttpContent(Unpooled.wrappedBuffer("abcd".getBytes())));
    int actuallyRead = entity.getContent().read(bytes);
    assertThat(actuallyRead, is(4));
    assertThat(new String(bytes), is("abcd"));

    handler.channelRead0(ctx, new DefaultHttpContent(Unpooled.wrappedBuffer("abcd".getBytes())));
    actuallyRead = entity.getContent().read(bytes);
    assertThat(actuallyRead, is(4));
    assertThat(new String(bytes), is("abcd"));

    handler.channelRead0(ctx, new DefaultHttpContent(Unpooled.wrappedBuffer("abcd".getBytes())));
    actuallyRead = entity.getContent().read(bytes);
    assertThat(actuallyRead, is(4));
    assertThat(new String(bytes), is("abcd"));

    handler.channelRead0(ctx, EMPTY_LAST_CONTENT);
    assertThat(entity.getContent().read(), is(-1));
  }

  @Test
  public void handleRequestAsFullRequest() throws Exception {
    DefaultHttpRequest request = new DefaultFullHttpRequest(
                                                            HTTP_1_1, POST, "https://www.mulesoft.com",
                                                            Unpooled.wrappedBuffer("abcd".getBytes()));
    request.headers().addInt(CONTENT_LENGTH, 4);

    handler.channelRead0(ctx, request);

    ArgumentCaptor<HttpRequestContext> reqCtxCaptor = forClass(HttpRequestContext.class);
    verify(requestHandler).handleRequest(reqCtxCaptor.capture(), any(HttpResponseReadyCallback.class));

    HttpRequest muleRequest = reqCtxCaptor.getValue().getRequest();
    HttpEntity entity = muleRequest.getEntity();

    byte[] bytes = new byte[4];
    int actuallyRead = entity.getContent().read(bytes);
    assertThat(actuallyRead, is(4));
    assertThat(new String(bytes), is("abcd"));

    assertThat(entity.getContent().read(), is(-1));
  }

  @Test
  // This can happen if the flow discards the payload without consuming it
  public void readerClosesTheStreamAndNothingFails() throws Exception {
    DefaultHttpRequest request = new DefaultHttpRequest(HTTP_1_1, POST, "https://www.mulesoft.com");
    request.headers().addInt(CONTENT_LENGTH, 12);

    handler.channelRead0(ctx, request);

    ArgumentCaptor<HttpRequestContext> reqCtxCaptor = forClass(HttpRequestContext.class);
    verify(requestHandler).handleRequest(reqCtxCaptor.capture(), any(HttpResponseReadyCallback.class));

    HttpRequest muleRequest = reqCtxCaptor.getValue().getRequest();
    HttpEntity entity = muleRequest.getEntity();

    byte[] bytes = new byte[4];

    handler.channelRead0(ctx, new DefaultHttpContent(Unpooled.wrappedBuffer("abcd".getBytes())));
    int actuallyRead = entity.getContent().read(bytes);
    assertThat(actuallyRead, is(4));
    assertThat(new String(bytes), is("abcd"));

    // The reader closed the pipe...
    entity.getContent().close();

    // And this is the call that shouldn't raise exceptions
    handler.channelRead0(ctx, new DefaultHttpContent(Unpooled.wrappedBuffer("abcd".getBytes())));
  }

  @Test
  public void errorHandler() throws Exception {
    DefaultHttpRequest request = new DefaultHttpRequest(HTTP_1_1, POST, "https://www.mulesoft.com");
    request.setDecoderResult(failure(new DecoderException("Expected")));

    handler.channelRead0(ctx, request);
    verify(errorRequestHandler).handleRequest(any(HttpRequestContext.class), any(HttpResponseReadyCallback.class));
  }
}
