/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static java.util.Collections.singleton;

import static javax.servlet.http.HttpServletResponse.SC_OK;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.utils.TestHttp2RequestHandler;
import org.mule.service.http.netty.utils.client.TestSSLNettyClientWithBouncyCastle;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.security.cert.CertificateException;

import io.netty.handler.ssl.ApplicationProtocolConfig;
import io.netty.handler.ssl.ApplicationProtocolConfig.Protocol;
import io.netty.handler.ssl.ApplicationProtocolConfig.SelectedListenerFailureBehavior;
import io.netty.handler.ssl.ApplicationProtocolConfig.SelectorFailureBehavior;
import io.netty.handler.ssl.ApplicationProtocolNames;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.IdentityCipherSuiteFilter;
import io.netty.handler.ssl.JdkSslContext;
import io.netty.handler.ssl.SslContext;
import io.qameta.allure.Issue;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class NettyHttpServerWithBouncyCastleTestCase extends AbstractMuleTestCase {

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");
  @Rule
  public TestSSLNettyClientWithBouncyCastle testClient =
      new TestSSLNettyClientWithBouncyCastle("localhost", serverPort.getNumber());
  private HttpServer serverWithSSLContext;
  private RequestHandlerManager requestHandlerManager;
  private HttpListenerRegistry listenerRegistry;

  @Before
  public void setup()
      throws IOException, CertificateException, NoSuchAlgorithmException, CreateException, KeyManagementException {
    Security.addProvider(new BouncyCastleProvider());
    listenerRegistry = new HttpListenerRegistry();
  }

  @After
  public void tearDown() {
    Security.removeProvider("BC");
    if (serverWithSSLContext != null) {
      serverWithSSLContext.stop().dispose();
    }
  }

  private void initiateServerWithSslContext(String keystorePath, String keystorePassword, String keyAlias, String truststorePath,
                                            String truststorePassword)
      throws NoSuchAlgorithmException, CreateException, KeyManagementException, IOException {

    SslContext serverSslContext =
        createSslContext(keystorePath, keystorePassword, keyAlias, truststorePath, truststorePassword);

    serverWithSSLContext = NettyHttpServer.builder()
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withSslContext(serverSslContext)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(
                                  new AcceptedConnectionChannelInitializer(listenerRegistry, true, 30000, serverSslContext, 300))
        .build();

    serverWithSSLContext.start();

    requestHandlerManager = serverWithSSLContext.addRequestHandler("/path", new TestHttp2RequestHandler());
    serverWithSSLContext.addRequestHandler(singleton("GET"), "/only-get", new TestHttp2RequestHandler());
  }

  private SslContext createSslContext(String keystorePath, String keystorePassword, String keyAlias, String truststorePath,
                                      String truststorePassword)
      throws NoSuchAlgorithmException, KeyManagementException, CreateException {

    TlsContextFactory tlsContextFactory =
        TlsContextFactory.builder().keyStorePath(keystorePath).keyStorePassword(keystorePassword).keyAlias(keyAlias)
            .keyPassword(keystorePassword).trustStorePath(truststorePath).trustStorePassword(truststorePassword)
            .insecureTrustStore(true).build();


    return new JdkSslContext(tlsContextFactory.createSslContext(), false, null, IdentityCipherSuiteFilter.INSTANCE,
                             new ApplicationProtocolConfig(Protocol.ALPN, SelectorFailureBehavior.NO_ADVERTISE,
                                                           SelectedListenerFailureBehavior.ACCEPT,
                                                           ApplicationProtocolNames.HTTP_2, ApplicationProtocolNames.HTTP_1_1),
                             ClientAuth.NONE, null, false);
  }

  @Test
  @Issue("W-15631497")
  public void testSSLConnection() throws Exception {
    initiateServerWithSslContext("serverKeystore", "mulepassword", "muleserver", "trustStore", "mulepassword");
    String sslEndpoint = "/path";
    HttpResponse response = testClient.sendGet(sslEndpoint);
    assertThat("Expected response status code to be 200", response.getStatusCode(), is(SC_OK));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLRehandshake() throws Exception {
    initiateServerWithSslContext("serverKeystore", "mulepassword", "muleserver", "trustStore", "mulepassword");
    String sslEndpoint = "/path";
    HttpResponse initialResponse = testClient.sendGet(sslEndpoint);
    assertThat(initialResponse.getStatusCode(), is(SC_OK));
    testClient.reHandshake();
    HttpResponse rehandshakeResponse = testClient.sendGet(sslEndpoint);
    assertThat(rehandshakeResponse.getStatusCode(), is(SC_OK));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLHandshakeFailureWhenKeystorePasswordIsIncorrect() {
    try {
      initiateServerWithSslContext("serverKeystore", "wrongpassword", "muleserver", "trustStore", "mulepassword");
      testClient.sendGet("/path");
    } catch (Exception e) {
      assertThat("Unable to initialise TLS configuration", is(e.getMessage()));
    }
  }

  @Test
  @Issue("W-15631497")
  public void testSSLHandshakeFailureWhenTrustStorePasswordIsIncorrect() {
    try {
      initiateServerWithSslContext("serverKeystore", "mulepassword", "muleserver", "trustStore", "wrongpassword");
      testClient.sendGet("/path");
    } catch (Exception e) {
      assertThat("Unable to initialise TLS configuration", is(e.getMessage()));
    }
  }
}
