/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static java.util.Collections.unmodifiableList;

import static io.netty.util.internal.logging.InternalLoggerFactory.getDefaultFactory;
import static io.netty.util.internal.logging.InternalLoggerFactory.setDefaultFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.netty.util.internal.logging.AbstractInternalLogger;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import org.apache.logging.log4j.message.FormattedMessageFactory;
import org.junit.rules.ExternalResource;

public class NettyLoggerRule extends ExternalResource {

  private InternalLoggerFactory defaultFactory;
  private TestLoggerFactory testFactory;

  @Override
  protected void before() throws Throwable {
    defaultFactory = getDefaultFactory();
    testFactory = new TestLoggerFactory(defaultFactory);
    setDefaultFactory(testFactory);
  }

  public List<String> getDebugLogs(String loggerName) {
    return testFactory.getLogger(loggerName).getDebugLogs();
  }

  @Override
  protected void after() {
    setDefaultFactory(defaultFactory);
  }

  private static class TestLoggerFactory extends InternalLoggerFactory {

    private final Map<String, TestInternalLogger> loggersByName = new HashMap<>();
    private final InternalLoggerFactory defaultFactory;

    public TestLoggerFactory(InternalLoggerFactory defaultFactory) {
      this.defaultFactory = defaultFactory;
    }

    @Override
    protected InternalLogger newInstance(String name) {
      return loggersByName.computeIfAbsent(name, key -> {
        setDefaultFactory(defaultFactory);
        try {
          InternalLogger actualNettyLogger = getInstance(key);
          return new TestInternalLogger(key, actualNettyLogger);
        } finally {
          setDefaultFactory(this);
        }
      });
    }

    public TestInternalLogger getLogger(String name) {
      return loggersByName.get(name);
    }
  }

  public static class TestInternalLogger extends InternalLoggerDelegate {

    private final FormattedMessageFactory messageFactory = new FormattedMessageFactory();
    private final List<String> debugLogs = new ArrayList<>();

    protected TestInternalLogger(String name, InternalLogger delegate) {
      super(name, delegate);
    }

    @Override
    public boolean isDebugEnabled() {
      // Always enabled for the test.
      return true;
    }

    @Override
    public void debug(String msg) {
      debugLogs.add(messageFactory.newMessage(msg).getFormattedMessage());
      super.debug(msg);
    }

    @Override
    public void debug(String fmt, Object arg1) {
      debugLogs.add(messageFactory.newMessage(fmt, arg1).getFormattedMessage());
      super.debug(fmt, arg1);
    }

    @Override
    public void debug(String fmt, Object arg1, Object arg2) {
      debugLogs.add(messageFactory.newMessage(fmt, arg1, arg2).getFormattedMessage());
      super.debug(fmt, arg1, arg2);
    }

    @Override
    public void debug(String fmt, Object... args) {
      debugLogs.add(messageFactory.newMessage(fmt, args).getFormattedMessage());
      super.debug(fmt, args);
    }

    @Override
    public void debug(String msg, Throwable throwable) {
      debugLogs.add(messageFactory.newMessage(msg, throwable).getFormattedMessage());
      super.debug(msg, throwable);
    }

    public List<String> getDebugLogs() {
      return unmodifiableList(debugLogs);
    }
  }

  public static class InternalLoggerDelegate extends AbstractInternalLogger {

    private final InternalLogger delegate;

    protected InternalLoggerDelegate(String name, InternalLogger delegate) {
      super(name);
      this.delegate = delegate;
    }

    @Override
    public boolean isDebugEnabled() {
      return delegate.isDebugEnabled();
    }

    @Override
    public void debug(String msg) {
      delegate.debug(msg);
    }

    @Override
    public void debug(String fmt, Object arg1) {
      delegate.debug(fmt, arg1);
    }

    @Override
    public void debug(String fmt, Object arg1, Object arg2) {
      delegate.debug(fmt, arg1, arg2);
    }

    @Override
    public void debug(String fmt, Object... args) {
      delegate.debug(fmt, args);
    }

    @Override
    public void debug(String msg, Throwable throwable) {
      delegate.debug(msg, throwable);
    }

    @Override
    public boolean isTraceEnabled() {
      return delegate.isTraceEnabled();
    }

    @Override
    public void trace(String s) {
      delegate.trace(s);
    }

    @Override
    public void trace(String s, Object o) {
      delegate.trace(s, o);
    }

    @Override
    public void trace(String s, Object o, Object o1) {
      delegate.trace(s, o, o1);
    }

    @Override
    public void trace(String s, Object... objects) {
      delegate.trace(s, objects);
    }

    @Override
    public void trace(String s, Throwable throwable) {
      delegate.trace(s, throwable);
    }

    @Override
    public boolean isInfoEnabled() {
      return delegate.isInfoEnabled();
    }

    @Override
    public void info(String s) {
      delegate.info(s);
    }

    @Override
    public void info(String s, Object o) {
      delegate.info(s, o);
    }

    @Override
    public void info(String s, Object o, Object o1) {
      delegate.info(s, o, o1);
    }

    @Override
    public void info(String s, Object... objects) {
      delegate.info(s, objects);
    }

    @Override
    public void info(String s, Throwable throwable) {
      delegate.info(s, throwable);
    }

    @Override
    public boolean isWarnEnabled() {
      return delegate.isWarnEnabled();
    }

    @Override
    public void warn(String s) {
      delegate.warn(s);
    }

    @Override
    public void warn(String s, Object o) {
      delegate.warn(s, o);
    }

    @Override
    public void warn(String s, Object... objects) {
      delegate.warn(s, objects);
    }

    @Override
    public void warn(String s, Object o, Object o1) {
      delegate.warn(s, o, o1);
    }

    @Override
    public void warn(String s, Throwable throwable) {
      delegate.warn(s, throwable);
    }

    @Override
    public boolean isErrorEnabled() {
      return delegate.isErrorEnabled();
    }

    @Override
    public void error(String s) {
      delegate.error(s);
    }

    @Override
    public void error(String s, Object o) {
      delegate.error(s, o);
    }

    @Override
    public void error(String s, Object o, Object o1) {
      delegate.error(s, o, o1);
    }

    @Override
    public void error(String s, Object... objects) {
      delegate.error(s, objects);
    }

    @Override
    public void error(String s, Throwable throwable) {
      delegate.error(s, throwable);
    }
  }
}
