/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NOT_MODIFIED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NO_CONTENT;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.RESET_CONTENT;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import static java.lang.Integer.getInteger;
import static java.lang.Long.parseLong;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.message.content.StreamedMultipartHttpEntity;

import java.io.IOException;
import java.io.InputStream;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponse;
import org.slf4j.Logger;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreator {

  private static final Logger LOGGER = getLogger(HttpResponseCreator.class);
  private static final String HEADER_CONTENT_TYPE = CONTENT_TYPE.toLowerCase();
  private static final String HEADER_CONTENT_LENGTH = CONTENT_LENGTH.toLowerCase();
  private static final int MAX_NUM_HEADERS_DEFAULT = 100;
  private static final String MAX_SERVER_RESPONSE_HEADERS_KEY = SYSTEM_PROPERTY_PREFIX + "http.MAX_SERVER_RESPONSE_HEADERS";
  private static int MAX_SERVER_RESPONSE_HEADERS =
      getInteger(MAX_SERVER_RESPONSE_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);


  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpResponse response, InputStream inputStream)
      throws IOException {

    HttpResponseBuilder responseBuilder = org.mule.runtime.http.api.domain.message.response.HttpResponse.builder()
        .statusCode(response.status().code())
        .reasonPhrase(response.status().reasonPhrase());

    // Header data...
    String contentType = null;
    String contentLength = null;
    HttpHeaders headers = response.headers();
    if (!headers.isEmpty()) {
      for (String headerName : headers.names()) {
        responseBuilder.addHeaders(headerName, response.headers().getAll(headerName));

        if (headerName.equalsIgnoreCase(HEADER_CONTENT_TYPE)) {
          contentType = response.headers().get(HEADER_CONTENT_TYPE);
        } else if (headerName.equalsIgnoreCase(HEADER_CONTENT_LENGTH)) {
          contentLength = response.headers().get(HEADER_CONTENT_LENGTH);
        }
      }
    }

    responseBuilder.entity(createEntity(inputStream, contentType, contentLength, response.status().code()));

    return responseBuilder.build();
  }

  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpClientResponse response,
                                                                               InputStream inputStream) {
    if (!checkMaxServerResponseHeadersLimit(response.responseHeaders())) {
      // todo: decide if we want to return a handler that throws 500 or another appropriate response
      throw new IllegalArgumentException("Exceeded max server response headers limit: " + MAX_SERVER_RESPONSE_HEADERS);
    }
    HttpResponseBuilder responseBuilder = org.mule.runtime.http.api.domain.message.response.HttpResponse.builder()
        .statusCode(response.status().code())
        .reasonPhrase(response.status().reasonPhrase());

    // Header data...
    String contentType = null;
    String contentLength = null;
    HttpHeaders headers = response.responseHeaders();
    if (!headers.isEmpty()) {
      for (String headerName : headers.names()) {
        responseBuilder.addHeaders(headerName, response.responseHeaders().getAll(headerName));

        if (headerName.equalsIgnoreCase(HEADER_CONTENT_TYPE)) {
          contentType = response.responseHeaders().get(HEADER_CONTENT_TYPE);
        } else if (headerName.equalsIgnoreCase(HEADER_CONTENT_LENGTH)) {
          contentLength = response.responseHeaders().get(HEADER_CONTENT_LENGTH);
        }
      }
    }

    responseBuilder.entity(createEntity(inputStream, contentType, contentLength, response.status().code()));

    return responseBuilder.build();
  }

  private static boolean checkMaxServerResponseHeadersLimit(HttpHeaders headers) {
    if (headers.size() > MAX_SERVER_RESPONSE_HEADERS) {
      LOGGER.warn("Exceeded max server response headers limit: {}. Current header count (including default headers): {}",
                  MAX_SERVER_RESPONSE_HEADERS,
                  headers.size());
      return false;
    }
    return true;
  }

  public static void refreshMaxServerResponseHeaders() {
    MAX_SERVER_RESPONSE_HEADERS = getInteger(MAX_SERVER_RESPONSE_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);
  }

  public static int getMaxServerResponseHeaders() {
    return MAX_SERVER_RESPONSE_HEADERS;
  }


  private HttpEntity createEntity(InputStream stream, String contentType, String contentLength, int statusCode) {
    long contentLengthAsLong = -1L;
    if (contentLength != null) {
      contentLengthAsLong = parseLong(contentLength);
    }
    if (contentType != null && contentType.startsWith(MULTIPART_MIXED.getPrimaryType())) {
      if (contentLengthAsLong >= 0) {
        return new StreamedMultipartHttpEntity(stream, contentType, contentLengthAsLong);
      } else {
        return new StreamedMultipartHttpEntity(stream, contentType);
      }
    } else {
      if (contentLengthAsLong > 0) {
        return new InputStreamHttpEntity(stream, contentLengthAsLong);
      } else if (contentLengthAsLong == 0) {
        return new EmptyHttpEntity();
      } else if (statusCode == NO_CONTENT.getStatusCode() || statusCode == NOT_MODIFIED.getStatusCode()
          || statusCode == RESET_CONTENT.getStatusCode()) {
        return new EmptyHttpEntity();
      } else {
        return new InputStreamHttpEntity(stream);
      }
    }
  }
}
