/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.HOST;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Names.USER_AGENT;
import static org.mule.runtime.http.api.HttpHeaders.Values.MULTIPART_FORM_DATA;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;

import static io.netty.handler.codec.http.HttpHeaderValues.CHUNKED;
import static io.netty.handler.codec.http.HttpResponseStatus.valueOf;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_0;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.OptionalLong;
import java.util.Set;

import io.netty.handler.codec.http.DefaultHttpResponse;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpVersion;

/**
 * Utility class that converts Mule objects to the Netty equivalent.
 */
public final class MuleToNettyUtils {

  private static final Set<String> DEFAULT_EMPTY_BODY_METHODS = unmodifiableSet(new HashSet<>(asList("GET", "HEAD", "OPTIONS")));

  private MuleToNettyUtils() {
    // private constructor to avoid wrong instantiations
  }

  /**
   * Converts the Mule representation of HTTP Response to the Netty one, only considering the header part (not the body).
   *
   * @param muleResponse Mule representation of HTTP Response.
   * @param httpProtocol
   * @return
   */
  public static io.netty.handler.codec.http.HttpResponse adaptResponseWithoutBody(
                                                                                  HttpResponse muleResponse,
                                                                                  HttpProtocol httpProtocol) {
    HttpVersion httpVersion = toNettyHttpVersion(httpProtocol);

    DefaultHttpResponse nettyResponse = new DefaultHttpResponse(requireNonNull(httpVersion),
                                                                valueOf(muleResponse.getStatusCode(),
                                                                        muleResponse.getReasonPhrase()));

    boolean hasTransferEncoding = false;
    boolean hasContentLength = false;
    MultiMap<String, String> headers = muleResponse.getHeaders();
    for (String key : headers.keySet()) {
      if (key.equalsIgnoreCase(HttpHeaderNames.CONTENT_LENGTH.toString())) {
        hasContentLength = true;
        nettyResponse.headers().setInt(key, Integer.parseInt(headers.get(key)));
      } else {
        nettyResponse.headers().set(key, sanitizeValues(headers.getAll(key)));
      }

      if (key.equalsIgnoreCase(HttpHeaderNames.TRANSFER_ENCODING.toString())) {
        hasTransferEncoding = true;
      }
    }

    if (hasContentLength || hasTransferEncoding) {
      return nettyResponse;
    }

    if (muleResponse.getEntity().getBytesLength().isPresent() && HTTP_1_1.equals(httpVersion)) {
      nettyResponse.headers().set(HttpHeaderNames.CONTENT_LENGTH, muleResponse.getEntity().getBytesLength().getAsLong());
    }
    return nettyResponse;
  }

  /**
   * Utility to transform the Mule HTTP API's {@link HttpProtocol} to the equivalent Netty's {@link HttpVersion}.
   *
   * @param protocol the Mule HTTP API value.
   * @return the Netty's value.
   */
  public static HttpVersion toNettyHttpVersion(HttpProtocol protocol) {
    switch (protocol) {
      case HTTP_0_9:
        return new HttpVersion("HTTP/0.9", false);
      case HTTP_1_0:
        return HTTP_1_0;
      case HTTP_1_1:
        return HTTP_1_1;
      default:
        return null;
    }
  }

  public static void addAllRequestHeaders(org.mule.runtime.http.api.domain.message.request.HttpRequest request,
                                          HttpHeaders currentHeaders, URI uri) {
    currentHeaders.set(HOST, buildHostHeaderValue(uri));
    currentHeaders.set(USER_AGENT, "Mule HTTP Client");

    boolean hasTransferEncoding = false;
    Collection<String> headerNames = request.getHeaderNames();
    for (String key : headerNames) {
      Collection<String> values = sanitizeValues(request.getHeaderValues(key));
      currentHeaders.set(key, values);
      if (key.equalsIgnoreCase(TRANSFER_ENCODING)) {
        hasTransferEncoding = true;
      }
    }

    org.mule.runtime.http.api.domain.HttpProtocol protocol = request.getProtocol();
    Object entity = request.getEntity();

    if (!hasTransferEncoding && protocol.equals(HttpProtocol.HTTP_1_1)
        && !(entity instanceof EmptyHttpEntity && DEFAULT_EMPTY_BODY_METHODS.contains(request.getMethod()))) {
      OptionalLong bytesLength = request.getEntity().getBytesLength();
      if (bytesLength.isPresent()) {
        currentHeaders.setInt(CONTENT_LENGTH, (int) bytesLength.getAsLong());
      } else {
        currentHeaders.set(TRANSFER_ENCODING, CHUNKED);
      }
    }

    if (request.getEntity().isComposed()) {
      currentHeaders.set(CONTENT_TYPE, MULTIPART_FORM_DATA);
    }
  }

  public static Collection<String> sanitizeValues(Collection<String> values) {
    boolean hasToSanitize = values.stream().anyMatch(value -> value != null && value.indexOf('\n') != -1);
    if (!hasToSanitize) {
      return values;
    }

    List<String> sanitized = new ArrayList<>(values.size());
    for (String value : values) {
      sanitized.add(sanitizeValue(value));
    }
    return sanitized;
  }

  public static String sanitizeValue(String value) {
    return (value != null) ? value.replace('\n', ' ') : null;
  }

  public static boolean calculateShouldRemoveContentLength(org.mule.runtime.http.api.domain.message.request.HttpRequest request) {
    return request.getEntity() instanceof EmptyHttpEntity && DEFAULT_EMPTY_BODY_METHODS.contains(request.getMethod());
  }

  private static int getDefaultPort(String scheme) {
    if (scheme == null || "http".equals(scheme)) {
      return 80;
    }
    if ("https".equals(scheme)) {
      return 443;
    }
    return -1;
  }

  private static String buildHostHeaderValue(URI uri) {
    String host = uri.getHost();
    if (host == null) {
      return "127.0.0.1";
    }
    int port = uri.getPort();
    int defaultPortForScheme = getDefaultPort(uri.getScheme());
    // If the port is default or not specified, simply return the host.
    return (port == defaultPortForScheme || port == -1) ? host : host + ":" + port;
  }
}
