/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.message;

import org.mule.service.http.netty.impl.client.auth.ntlm.NtlmFlags;

/**
 * Represents a Type 1 NTLM message used in NTLM authentication.
 * <p>
 * This message is sent by the client to initiate the NTLM authentication process. It contains information such as negotiation
 * flags and other details necessary for the server to respond appropriately.
 * </p>
 * <p>
 * This implementation is based on the jcifs library, available at <a href="https://github.com/kevinboone/jcifs">jcifs GitHub
 * Repository</a>.
 * </p>
 */
public class Type1Message extends NtlmMessage {

  protected static final byte[] NTLMSSP_SIGNATURE = new byte[] {
      (byte) 'N', (byte) 'T', (byte) 'L', (byte) 'M', (byte) 'S', (byte) 'S', (byte) 'P', (byte) 0
  };

  public Type1Message(int flags) {
    super(flags);
  }

  public byte[] toByteArray() {
    int flags = getFlags();
    int size = 8 * 4 + ((flags & NtlmFlags.NTLMSSP_NEGOTIATE_VERSION) != 0 ? 8 : 0);

    byte[] domain = new byte[0];
    byte[] workstation = new byte[0];

    byte[] type1 = new byte[size];
    int pos = 0;

    System.arraycopy(NTLMSSP_SIGNATURE, 0, type1, 0, NTLMSSP_SIGNATURE.length);
    pos += NTLMSSP_SIGNATURE.length;

    writeULong(type1, pos, NTLMSSP_TYPE1);
    pos += 4;

    writeULong(type1, pos, flags);
    pos += 4;

    int domOffOff = writeSecurityBuffer(type1, pos, domain);
    pos += 8;

    int wsOffOff = writeSecurityBuffer(type1, pos, workstation);
    pos += 8;

    if ((flags & NtlmFlags.NTLMSSP_NEGOTIATE_VERSION) != 0) {
      System.arraycopy(NTLMSSP_VERSION, 0, type1, pos, NTLMSSP_VERSION.length);
      pos += NTLMSSP_VERSION.length;
    }

    pos += writeSecurityBufferContent(type1, pos, domOffOff, domain);
    writeSecurityBufferContent(type1, pos, wsOffOff, workstation);
    return type1;
  }
}
