/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.server.sse;

import org.mule.runtime.core.api.util.UUID;
import org.mule.runtime.http.api.sse.server.SseClient;
import org.mule.runtime.http.api.sse.server.SseClientConfig;
import org.mule.service.http.common.message.sse.ServerSentEventImpl;
import org.mule.service.http.common.message.sse.SseEntityEncoder;

import java.io.IOException;
import java.io.Writer;
import java.net.InetSocketAddress;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * Basic implementation of {@link SseClient}.
 */
public class SseClientImpl implements SseClient {

  private final Writer bodyWriter;
  private final SseEntityEncoder encoder;
  private final String clientId;
  private final InetSocketAddress remoteAddress;
  private CompletableFuture<Void> responseSentFuture;

  public SseClientImpl(SseClientConfig config, Writer bodyWriter, CompletableFuture<Void> responseSentFuture) {
    this.bodyWriter = bodyWriter;
    this.responseSentFuture = responseSentFuture;
    this.encoder = new SseEntityEncoder();
    this.clientId = config.getClientId() != null ? config.getClientId() : createUUID();
    this.remoteAddress = config.getRemoteHostAddress();
  }

  @Override
  public void sendEvent(String eventName, String payload, String id, Long retryDelay) throws IOException {
    encoder.writeTo(bodyWriter, new ServerSentEventImpl(eventName, payload, id, retryDelay));
  }

  @Override
  public void sendComment(String comment) {
    // TODO (W-18041205): Implement comments.
  }

  @Override
  public void onClose(Consumer<Throwable> callback) {
    responseSentFuture = responseSentFuture.whenComplete((ignored, error) -> callback.accept(error));
  }

  @Override
  public void close() throws IOException {
    bodyWriter.close();
  }

  @Override
  public String getClientId() {
    return clientId;
  }

  @Override
  public InetSocketAddress getRemoteAddress() {
    return remoteAddress;
  }

  private String createUUID() {
    // TODO (W-18041272): Make it cluster-aware.
    return UUID.getUUID();
  }
}
