/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.benchmark;


import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.ByteArrayInputStream;
import java.util.concurrent.TimeUnit;

import io.qameta.allure.Issue;
import org.junit.Rule;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.infra.Blackhole;

@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
@State(Scope.Thread)
public class BenchmarkTest {

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");
  private BenchmarkTestServer testServer;
  private HttpClient client;
  private String testUrl;

  private HttpRequest prebuiltGetRequest;
  private HttpRequest prebuiltPostNonStreamingRequest;
  private HttpRequest prebuiltPostStreamingRequest;


  @Setup(Level.Trial)
  public void setUp() throws Exception {
    testServer = new BenchmarkTestServer("localhost", serverPort.getNumber());

    testServer.addRequestHandler("/hello", (request, responseSender) -> {
      HttpEntity helloContent = new StringHttpEntity("Hello from benchmark server!");
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();


    client = NettyHttpClient.builder().withConnectionIdleTimeout(1000).withUsingPersistentConnections(true).build();
    client.start();

    testUrl = format("http://localhost:%d/hello", serverPort.getNumber());
    buildRequests();

  }

  @TearDown(Level.Trial)
  public void tearDown() {
    if (client != null) {
      client.stop();
    }
    if (testServer != null) {
      testServer.stop();
    }
  }

  @Issue("W-17282518")
  @Benchmark
  public void testHttpGetRequest(Blackhole blackhole) {
    try {
      HttpResponse response = client.sendAsync(prebuiltGetRequest).get();
      blackhole.consume(response.getStatusCode());
      blackhole.consume(response.getEntity());
    } catch (Exception e) {
      throw new RuntimeException("Request execution failed", e);
    }
  }

  @Issue("W-17363023")
  @Benchmark
  public void testHttpPostWithNonStreamingRequest(Blackhole blackhole) {
    try {
      HttpResponse response = client.sendAsync(prebuiltPostNonStreamingRequest).get();
      blackhole.consume(response.getStatusCode());
      blackhole.consume(response.getEntity());
    } catch (Exception e) {
      throw new RuntimeException("Request execution failed", e);
    }
  }

  @Issue("W-17363023")
  @Benchmark
  public void testHttpPostWithStreamingRequest(Blackhole blackhole) {
    try {
      HttpResponse response = client.sendAsync(prebuiltPostStreamingRequest).get();
      blackhole.consume(response.getStatusCode());
      blackhole.consume(response.getEntity());
    } catch (Exception e) {
      throw new RuntimeException("Request execution failed", e);
    }
  }

  private void buildRequests() {
    prebuiltGetRequest = HttpRequest.builder().uri(testUrl).method("GET").entity(new EmptyHttpEntity()).build();
    prebuiltPostNonStreamingRequest =
        HttpRequest.builder().uri(testUrl).method("POST").entity(new StringHttpEntity("Hello from benchmarking client!")).build();
    String message = "Hello from benchmarking client!";
    prebuiltPostStreamingRequest = HttpRequest.builder().uri(testUrl).method("POST")
        .entity(new InputStreamHttpEntity(new ByteArrayInputStream(message.getBytes(UTF_8)))).build();
  }
}

