/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;

import static java.lang.Integer.parseInt;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static io.netty.handler.codec.http.HttpObjectDecoder.DEFAULT_MAX_CHUNK_SIZE;
import static io.netty.handler.codec.http.HttpObjectDecoder.DEFAULT_MAX_HEADER_SIZE;
import static io.netty.handler.flush.FlushConsolidationHandler.DEFAULT_EXPLICIT_FLUSH_AFTER_FLUSHES;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.util.HttpLoggingHandler;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import io.netty.channel.Channel;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelPipeline;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.codec.http.HttpServerCodec;
import io.netty.handler.flush.FlushConsolidationHandler;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.timeout.IdleStateHandler;

/**
 * Implementation of {@link ChannelInitializer} that handles SSL if a non-null {@link SslContext} is received, and after that adds
 * the needed decoders ending by delegate to the passed {@link HttpListenerRegistry}.
 */
public class AcceptedConnectionChannelInitializer extends ChannelInitializer<SocketChannel> {

  // Defines the maximum size in bytes accepted for the http request header section (request line + headers)
  public static final String MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY = SYSTEM_PROPERTY_PREFIX + "http.headerSectionSize";
  public static final String FLUSH_CONSOLIDATION_HANDLER_NAME = "flushConsolidationHandler";

  private final SslContext sslContext;
  private final HttpListenerRegistry httpListenerRegistry;
  private final boolean usePersistentConnections;
  private final long connectionIdleTimeout;
  private final long readTimeout;

  private final int maxInitialLineLength;
  private final int maxHeaderSize;

  private final ConnectionsCounterHandler connectionsCountHandler;
  private final ExecutorService ioExecutor;

  public AcceptedConnectionChannelInitializer(HttpListenerRegistry httpListenerRegistry,
                                              HttpServerConfiguration configuration,
                                              SslContext sslContext,
                                              ExecutorService ioExecutor) {
    this(httpListenerRegistry, configuration.isUsePersistentConnections(), configuration.getConnectionIdleTimeout(),
         configuration.getReadTimeout(), sslContext, ioExecutor);
  }

  public AcceptedConnectionChannelInitializer(HttpListenerRegistry httpListenerRegistry,
                                              boolean usePersistentConnections,
                                              int connectionIdleTimeout,
                                              long readTimeout,
                                              SslContext sslContext,
                                              ExecutorService ioExecutor) {
    this(httpListenerRegistry, usePersistentConnections, connectionIdleTimeout, readTimeout, sslContext,
         retrieveMaximumHeaderSectionSize(), ioExecutor);
  }

  public AcceptedConnectionChannelInitializer(HttpListenerRegistry httpListenerRegistry,
                                              boolean usePersistentConnections,
                                              int connectionIdleTimeout,
                                              long readTimeout,
                                              SslContext sslContext,
                                              int maxHeaderSectionSize,
                                              ExecutorService ioExecutor) {
    this.httpListenerRegistry = httpListenerRegistry;
    this.usePersistentConnections = usePersistentConnections;
    this.connectionIdleTimeout = connectionIdleTimeout;
    this.readTimeout = readTimeout;
    this.sslContext = sslContext;

    this.maxInitialLineLength = maxHeaderSectionSize;
    this.maxHeaderSize = maxHeaderSectionSize;

    this.connectionsCountHandler = new ConnectionsCounterHandler();
    this.ioExecutor = ioExecutor;
  }

  @Override
  protected void initChannel(SocketChannel socketChannel) {
    socketChannel.pipeline().addFirst(FLUSH_CONSOLIDATION_HANDLER_NAME,
                                      new FlushConsolidationHandler(DEFAULT_EXPLICIT_FLUSH_AFTER_FLUSHES, true));
    socketChannel.pipeline().addFirst(connectionsCountHandler);

    configureTimeouts(socketChannel);

    if (null != sslContext) {
      configureWithSslAndAPN(socketChannel);
    } else {
      configureHttp1(socketChannel.pipeline(), null);
    }
  }

  /**
   * Waits for all the in-flight connections to be closed, for the time passed as parameter.
   * 
   * @param timeout  time to wait. Zero means don't wait.
   * @param timeUnit unit for the timeout parameter.
   */
  public void waitForConnectionsToBeClosed(Long timeout, TimeUnit timeUnit) {
    connectionsCountHandler.waitForConnectionsToBeClosed(timeout, timeUnit);
  }

  /**
   * Configure the pipeline for TLS NPN negotiation to HTTP/1.
   */
  private void configureWithSslAndAPN(Channel channel) {
    SslHandler sslHandler = sslContext.newHandler(channel.alloc());
    channel.pipeline()
        .addLast("SSL Handler", sslHandler)
        .addLast("Protocol Negotiation Handler", new HttpWithAPNServerHandler(this::configureHttp1, sslHandler));
  }

  private void configureTimeouts(SocketChannel socketChannel) {
    if (connectionIdleTimeout != -1 || readTimeout != -1) {
      socketChannel.pipeline().addLast("idleStateHandler",
                                       new IdleStateHandler(readTimeout, connectionIdleTimeout, connectionIdleTimeout,
                                                            MILLISECONDS));
    }
  }

  protected void configureHttp1(ChannelPipeline pipeline, SslHandler sslHandler) {
    pipeline.addLast("Logging Handler", new HttpLoggingHandler());
    pipeline.addLast("HTTP/1 Codec", new HttpServerCodec(maxInitialLineLength, maxHeaderSize, DEFAULT_MAX_CHUNK_SIZE));
    pipeline.addLast("Expect Continue Handler", new MuleHttpServerExpectContinueHandler());
    pipeline.addLast("Keep Alive", new KeepAliveHandler(usePersistentConnections));
    pipeline.addLast("Forward Initializer", new ForwardingToListenerInitializer(httpListenerRegistry, sslHandler, ioExecutor));
  }

  private static int retrieveMaximumHeaderSectionSize() {
    try {
      return parseInt(getProperty(MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY, String.valueOf(DEFAULT_MAX_HEADER_SIZE)));
    } catch (NumberFormatException e) {
      throw new MuleRuntimeException(createStaticMessage(format("Invalid value %s for %s configuration",
                                                                getProperty(MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY),
                                                                MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY)),
                                     e);
    }
  }
}
