/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static org.mule.runtime.http.api.server.HttpServerProperties.refreshSystemProperties;

import static java.lang.System.nanoTime;

import static org.junit.Assert.fail;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.core.api.util.func.CheckedRunnable;

import java.io.IOException;
import java.net.Socket;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import io.netty.handler.ssl.ApplicationProtocolConfig;
import io.netty.handler.ssl.ApplicationProtocolNames;
import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.IdentityCipherSuiteFilter;
import io.netty.handler.ssl.JdkSslContext;
import io.netty.handler.ssl.SslContext;
import org.junit.function.ThrowingRunnable;

public final class TestUtils {

  private TestUtils() {
    // prevent instantiation
  }

  /**
   * Utility method that sets the property PRESERVE_HEADER_CASE before running the passed callback, and clears it after.
   */
  public static void preservingHeaderCase(CheckedRunnable runnable) {
    System.setProperty("org.glassfish.grizzly.http.PRESERVE_HEADER_CASE", "true");
    refreshSystemProperties();
    try {
      runnable.run();
    } finally {
      System.clearProperty("org.glassfish.grizzly.http.PRESERVE_HEADER_CASE");
      refreshSystemProperties();
    }
  }

  /**
   * Utility method to send a raw String to the server and receive the raw response. It's useful for testing some specific
   * scenarios when you matter exactly what String is going through the socket. WARNING: This method doesn't take care about
   * short-reads and short-writes, it's just an utility method for simple tests.
   */
  public static String sendRawRequestToServer(String rawRequestToSend, String host, int port) throws IOException {
    try (Socket clientSocket = new Socket(host, port)) {
      clientSocket.getOutputStream().write(rawRequestToSend.getBytes());

      byte[] responseBuffer = new byte[1024];

      // Assuming that the response fits in the buffer, and that it will be read in only one reading
      clientSocket.getInputStream().read(responseBuffer);
      return new String(responseBuffer);
    }
  }

  /**
   * Creates a {@link SslContext} for server-side usage in tests.
   * 
   * @return a {@link SslContext}.
   */
  public static SslContext createServerSslContext() throws NoSuchAlgorithmException, KeyManagementException, CreateException {
    TlsContextFactory tlsContextFactory =
        TlsContextFactory.builder().enabledCipherSuites("TLS_SOMETHING").enabledProtocols("TLSv1.1")
            .keyStorePath("serverKeystore")
            .keyStorePassword("mulepassword").keyAlias("muleserver").keyPassword("mulepassword").keyStoreAlgorithm("PKIX")
            .trustStorePath("trustStore").trustStorePassword("mulepassword").trustStoreType("jceks").insecureTrustStore(true)
            .build();
    return new JdkSslContext(tlsContextFactory.createSslContext(), false, null, IdentityCipherSuiteFilter.INSTANCE,
                             new ApplicationProtocolConfig(ApplicationProtocolConfig.Protocol.ALPN,
                                                           // NO_ADVERTISE is currently the only mode supported by both
                                                           // OpenSsl and JDK providers.
                                                           ApplicationProtocolConfig.SelectorFailureBehavior.NO_ADVERTISE,
                                                           // ACCEPT is currently the only mode supported by both OpenSsl
                                                           // and JDK providers.
                                                           ApplicationProtocolConfig.SelectedListenerFailureBehavior.ACCEPT,
                                                           ApplicationProtocolNames.HTTP_2, ApplicationProtocolNames.HTTP_1_1),
                             ClientAuth.NONE, null, false);
  }

  /**
   * Executes a callback measuring the elapsed nanoseconds. If the callback throws an exception, it will make the test fail.
   * 
   * @param runnable the callback to be executed.
   * @return the elapsed time in nanoseconds.
   */
  public static long measuringNanoseconds(ThrowingRunnable runnable) {
    long nanosBefore = nanoTime();
    try {
      runnable.run();
    } catch (Throwable t) {
      fail("Unexpected exception: " + t);
    }
    long nanosAfter = nanoTime();
    return (nanosAfter - nanosBefore);
  }
}
