/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.service;

import static org.mule.runtime.api.artifact.ArtifactType.APP;
import static org.mule.runtime.api.artifact.ArtifactType.DOMAIN;
import static org.mule.runtime.api.artifact.ArtifactType.POLICY;
import static org.mule.runtime.core.api.config.MuleProperties.APP_NAME_PROPERTY;
import static org.mule.runtime.core.api.config.MuleProperties.DOMAIN_NAME_PROPERTY;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.artifact.ArtifactType;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientFactory;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.runtime.http.api.utils.RequestMatcherRegistry;
import org.mule.service.http.netty.impl.client.HttpClientConnectionManager;
import org.mule.service.http.netty.impl.server.HttpServerConnectionManager;
import org.mule.service.http.netty.impl.server.util.DefaultRequestMatcherRegistryBuilder;

import java.util.Optional;

import javax.inject.Inject;

import org.slf4j.Logger;

/**
 * Implementation of {@link HttpService} that uses Netty to create {@link HttpServer}s and Reactor Netty to create
 * {@link HttpClient}s.
 */
public class NettyHttpServiceImplementation implements HttpService, Startable, Stoppable {

  private final HttpServerConnectionManager serverConnectionManager;
  private final HttpClientConnectionManager clientConnectionManager;

  private static final Logger LOGGER = getLogger(NettyHttpServiceImplementation.class);

  public NettyHttpServiceImplementation(SchedulerService schedulerService) {
    this(schedulerService, new HttpServerConnectionManager(schedulerService), new HttpClientConnectionManager(schedulerService));
  }

  protected NettyHttpServiceImplementation(SchedulerService schedulerService,
                                           HttpServerConnectionManager serverConnectionManager,
                                           HttpClientConnectionManager clientConnectionManager) {
    this.serverConnectionManager = serverConnectionManager;
    this.clientConnectionManager = clientConnectionManager;
  }

  @Override
  public HttpServerFactory getServerFactory() {
    return new ContextHttpServerFactory(serverConnectionManager, "container", null, () -> 1000L);
  }

  @Inject
  public HttpServerFactory getServerFactory(MuleContext muleContext, Registry registry) {
    Optional<String> appName = registry.lookupByName(APP_NAME_PROPERTY);
    Optional<String> domainName = registry.lookupByName(DOMAIN_NAME_PROPERTY);
    ArtifactType artifactType = muleContext.getArtifactType().getArtifactType();
    String systemName = muleContext.getId();
    String currentCtxId = getCurrentContextName(artifactType, appName.orElse(null), domainName.orElse(null), systemName);
    String parentCtxId = getParentContextName(artifactType, domainName.orElse(null));
    return new ContextHttpServerFactory(serverConnectionManager, currentCtxId, parentCtxId,
                                        () -> muleContext.getConfiguration().getShutdownTimeout());
  }

  private String getParentContextName(ArtifactType artifactType, String domainName) {
    if (APP == artifactType && domainName != null) {
      return domainName + "-" + DOMAIN.getArtifactTypeAsString();
    }
    return null;
  }

  private String getCurrentContextName(ArtifactType artifactType, String appName, String domainName, String systemName) {
    if (appName == null && domainName == null) {
      return systemName;
    }
    if (APP == artifactType || POLICY == artifactType) {
      return appName + "-" + artifactType.getArtifactTypeAsString();
    }
    if (DOMAIN == artifactType) {
      return domainName + "-" + DOMAIN.getArtifactTypeAsString();
    }
    return systemName;
  }

  @Override
  public HttpClientFactory getClientFactory() {
    return new ContextHttpClientFactory(clientConnectionManager);
  }

  @Override
  public RequestMatcherRegistry.RequestMatcherRegistryBuilder getRequestMatcherRegistryBuilder() {
    return new DefaultRequestMatcherRegistryBuilder();
  }

  @Override
  public String getName() {
    return "HTTP Service using Netty";
  }

  @Override
  public void start() throws MuleException {
    initialiseIfNeeded(serverConnectionManager);
    initialiseIfNeeded(clientConnectionManager);
  }

  @Override
  public void stop() throws MuleException {
    disposeIfNeeded(serverConnectionManager, LOGGER);
    disposeIfNeeded(clientConnectionManager, LOGGER);
  }
}
